/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file CovPoseTest.C
 *
 * @author Erik Einhorn
 * @date   2011/07/07
 */

#include <fw/Unit.h>
#include <transform/Pose.h>

namespace mira { namespace test {

///////////////////////////////////////////////////////////////////////////////

class CovPoseTest : public Unit
{
	MIRA_OBJECT(CovPoseTest)
public:
	CovPoseTest() :
		Unit(Duration::milliseconds(500))
	{
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		Unit::reflect(r);
	}

	virtual void initialize()
	{
		mPose2CovBase = publish<PoseCov2>("Pose2CovBase");
		mPose2Cov1 = publish<PoseCov2>("Pose2Cov1");
		mPose2CovV = publish<Pose2CovVector>("Pose2CovVector");
		mPose2Cov2 = publish<PoseCov2>("Pose2Cov2");

		PoseCov2 p2;
		p2.t = Eigen::Vector2f(-2,0);
		p2.r = deg2rad(45.0f);
		p2.cov << 0.5f,0,0,
				 0,0.1f,0,
				 0,0,0.001f;

		addTransformLink("Pose2CovBase","Root");
		mPose2CovBase.post(p2);
		addTransformLink("Pose2Cov1","Pose2CovBase");

		p2.t = Eigen::Vector2f(2,0);
		p2.r = 0.0f;

		p2.cov << 1,0,0,
			 	 0,0.1f,0,
				 0,0,0.05f;

		addTransformLink("Pose2Cov2","Pose2Cov1");
		mPose2Cov2.post(p2);

		p2.t = Eigen::Vector2f(-2,0);
		auto v = mPose2CovV.write();
		v->push_back(p2);
		v->timestamp = Time::now();
		v->frameID = resolveName("Pose2Cov1");

		///////////////////////////////////////////////////////////////////////

		mPose3CovBase = publish<PoseCov3>("Pose3CovBase");
		mPose3CovV = publish<Pose3CovVector>("Pose3CovVector");
		mPose3Cov1 = publish<PoseCov3>("Pose3Cov1");
		mPose3Cov2 = publish<PoseCov3>("Pose3Cov2");

		PoseCov3 p3;
		p3.t = Eigen::Vector3f(2,0,0);
		p3.r = Eigen::Quaternionf(Eigen::AngleAxisf(deg2rad(45.0f), Eigen::Vector3f::UnitZ()));
		p3.cov << 0.5f,0,0,  0,0,0,0,
			 	  0,0.1f,0,  0,0,0,0,
				  0,0,0.05f, 0,0,0,0,
				  0,0,0,     0.001f,0,0,0,
				  0,0,0,     0,0.001f,0,0,
				  0,0,0,     0,0,0.001f,0,
				  0,0,0,     0,0,0,0.001f;

		addTransformLink("Pose3CovBase","Root");
		mPose3CovBase.post(p3);
		addTransformLink("Pose3Cov1","Pose3CovBase");


		p3.t = Eigen::Vector3f(2,0,0);
		p3.r = Eigen::Quaternionf(Eigen::AngleAxisf(deg2rad(0.0f), Eigen::Vector3f::UnitZ()));
		p3.cov << 1,0,0,     0,0,0,0,
			 	  0,0.1f,0,  0,0,0,0,
				  0,0,0.05f, 0,0,0,0,
				  0,0,0,     0.05f,0,0,0,
				  0,0,0,     0,0.05f,0,0,
				  0,0,0,     0,0,0.05f,0,
				  0,0,0,     0,0,0,0.05f;

		addTransformLink("Pose3Cov2","Pose3Cov1");
		mPose3Cov2.post(p3);

		p3.t = Eigen::Vector3f(-2,0,0);
		auto v3 = mPose3CovV.write();
		v3->push_back(p3);
		v3->timestamp = Time::now();
		v3->frameID = resolveName("Pose3Cov1");



		p3.t = Eigen::Vector3f(0,1,1);
		p3.r = Eigen::Quaternionf(Eigen::AngleAxisf(deg2rad(45.0f), Eigen::Vector3f::UnitX()));
		mPose3Cov1.post(p3);
	}

	virtual void process(const Timer& timer)
	{
		static float phi=0.0f;

		PoseCov2 p2;
		p2.t = Eigen::Vector2f(0,1);
		p2.r = phi;
		mPose2Cov1.post(p2);

		//PoseCov3 p3;
		/*p3.t = Eigen::Vector3f(0,1,1);
		p3.r = Eigen::Quaternionf(Eigen::AngleAxisf(phi, Eigen::Vector3f::UnitZ()));
		mPose3Cov1.post(p3);*/

		/*p3.t = Eigen::Vector3f(2,0,0);
		p3.r = Eigen::Quaternionf(Eigen::AngleAxisf(phi, Eigen::Vector3f::UnitZ()));
		p3.cov << 1,0,0,    0,0,0,
			 	  0,0.1,0,  0,0,0,
				  0,0,0.05, 0,0,0,
				  0,0,0,    0.05,0,0,
				  0,0,0,    0,0.05,0,
				  0,0,0,    0,0,0.05;
*/

		Eigen::Matrix<float,6,6> cov;
		cov  << 3,0,0,       0,0,0,
			 	  0,0.1f,0,  0,0,0,
				  0,0,0.05f, 0,0,0,
				  0,0,0,     0.05f,0,0,
				  0,0,0,     0,0.05f,0,
				  0,0,0,     0,0,0.05f;
		Eigen::Quaternionf orientation = Eigen::Quaternionf(Eigen::AngleAxisf(phi, Eigen::Vector3f::UnitZ()));
		//std::cout<<"CovPoseTest: euler cov = "<<std::endl<<print(cov)<<std::endl;

		PoseCov3 p3(Eigen::Vector3f(-2,0,0), orientation, quaternionCovFromYawPitchRollCov(cov, orientation));
		//std::cout<<"CovPoseTest: quat cov = "<<std::endl<<print(p3.cov)<<std::endl;

		//std::cout<<"CovPoseTest: euler cov test= "<<std::endl<<print(quaternionCovToYawPitchRollCov(p3.cov, orientation))<<std::endl;
		mPose3Cov1.post(p3);

		orientation = (Eigen::AngleAxisf(phi, Eigen::Vector3f::UnitZ()));
		PoseCov3 p3b(Eigen::Vector3f(2,0,0), orientation, quaternionCovFromYawPitchRollCov(cov, orientation));
		auto v3 = mPose3CovV.write();
		v3->clear();
		v3->push_back(p3b);
		v3->timestamp = timer.current;
		v3->frameID = resolveName("Pose3Cov1");


		phi+=0.05f;
	}

protected:
	Channel<PoseCov2> mPose2CovBase;
	Channel<PoseCov2> mPose2Cov1;
	Channel<PoseCov2> mPose2Cov2;
	Channel<Pose2CovVector> mPose2CovV;

	Channel<PoseCov3> mPose3CovBase;
	Channel<PoseCov3> mPose3Cov1;
	Channel<PoseCov3> mPose3Cov2;
	Channel<Pose3CovVector> mPose3CovV;

	int mCounter;
};

///////////////////////////////////////////////////////////////////////////////

}}

MIRA_CLASS_SERIALIZATION(mira::test::CovPoseTest, mira::Unit);
