/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file FrameworkTransformTest.C
 *
 * @author Erik Einhorn
 * @date   2010/12/06
 */

#define ALLOWED_DELAY 10 // in ms
#define ALLOWED_RPC_DELAY 20 // in ms

#include <boost/test/unit_test.hpp>

#include <fw/Framework.h>
#include <math/LinearInterpolator.h>
#include <serialization/JSONSerializer.h>

#include <transform/RigidTransform.h>

using namespace mira;
using namespace std;

//const char* argv[] = {"FrameworkTransformTest", "-d5", "--no-colors"};
const char* argv[] = {"FrameworkTransformTest", "-d0", "--no-colors"};
Framework fw(3,(char**)argv);

///////////////////////////////////////////////////////////////////////////////

/*
Tests:
- transform tree
- isTransformAvailable
- waitForTransform
- concatenation of transform within transform tree
- linear interpolation of transform within transform tree
 */
BOOST_AUTO_TEST_CASE(InterpolationTests)
{
	Authority authority("/InterpolationTests");

	Time t0  = Time::now();

	Time tm1 = t0-Duration::seconds(1);
	//   t0
	Time t1   = t0+Duration::seconds(1);
	Time t1s  = t1+Duration::microseconds(1);

	/*
	t0-1: (explicitly given)

				 B --> C(2,1)
				 ^
				 |
		   R --> A


	t0:  (not given, interpolated)

		   R --> A ------> B --> C(4,0)

	t0+1: (explicitly given)

		   R --> A
				 |
				 |
				 |
				 |
				 v
				 B --> C(2,-3)
	y
	^
	|
	--->x
	*/

	RigidTransform2f Am1(1.0f, 0.0f, Degreef(89.99f));
	RigidTransform2f A1 (1.0f, 0.0f, Degreef(-89.99f));

	RigidTransform2f Bm1(1.0f, 0.0f, Degreef(-89.99f));
	RigidTransform2f B1 (3.0f, 0.0f, Degreef( 89.99f));

	RigidTransform2f C(1.0f, 0.0f, 0.0f);

	authority.addTransformLink("A", "R");
	authority.addTransformLink("B", "A");
	authority.addTransformLink("C", "B");



	BOOST_CHECK(!authority.isTransformAvailable("C", "R"));
	BOOST_CHECK(!authority.waitForTransform("C", "R", Duration::seconds(1)));

	authority.publishTransform("A", Am1, tm1);
	MIRA_FW.getChannelManager().setStorageDuration("/A",Duration::seconds(10));
	authority.publishTransform("A", A1 , t1);
	authority.publishTransform("A", A1 , t1s);

	authority.publishTransform("B", Bm1, tm1);
	authority.publishTransform("B", B1 , t1);

	authority.publishTransform("C", C, tm1);

	BOOST_CHECK(authority.isTransformAvailable("C", "R"));
	BOOST_CHECK(authority.waitForTransform("C", "R", Duration::seconds(1)));

	RigidTransform2f rm1 = authority.getTransform<RigidTransform2f>("C","R", tm1+Duration::milliseconds(1));
	cout << "rm1:" << print(rm1) << endl;

	BOOST_CHECK_CLOSE(rm1.x(),   2.0f, 0.1f);
	BOOST_CHECK_CLOSE(rm1.y(),   1.0f, 0.1f);
	BOOST_CHECK_SMALL(Anglef(rm1.phi()).smallestDifferenceValue(Anglef(0.0f)), 0.1f);

	RigidTransform2f r1 = authority.getTransform<RigidTransform2f>("C","R", t1+Duration::milliseconds(1));
	cout << "r1:" << print(r1) << endl;

	BOOST_CHECK_CLOSE(r1.x(),   2.0f, 0.1f);
	BOOST_CHECK_CLOSE(r1.y(),  -3.0f, 0.1f);
	BOOST_CHECK_SMALL(Anglef(r1.phi()).smallestDifferenceValue(Anglef(0.0f)), 0.1f);

	RigidTransform2f r0i = authority.getTransform<RigidTransform2f>("C","R", t0, LinearInterpolator());
	cout << "r0:" << print(r0i) << endl;

	BOOST_CHECK_CLOSE(r0i.x(),   4.0f, 0.1f);
	BOOST_CHECK_SMALL(r0i.y(),         0.1f);
	BOOST_CHECK_SMALL(Anglef(r0i.phi()).smallestDifferenceValue(Anglef(0.0f)), 0.1f);

	RigidTransform2f r1i = authority.getTransform<RigidTransform2f>("C","R", t1, LinearInterpolator());
	cout << "r1i:" << print(r1i) << endl;

	BOOST_CHECK_CLOSE(r1i.x(),   2.0f, 0.1f);
	BOOST_CHECK_CLOSE(r1i.y(),  -3.0f, 0.1f);
	BOOST_CHECK_SMALL(Anglef(r1i.phi()).smallestDifferenceValue(Anglef(0.0f)), 0.1f);


	RigidTransform2f r1si = authority.getTransform<RigidTransform2f>("C","R", t1s, LinearInterpolator());
	cout << "r1si:" << print(r1si) << endl;

	BOOST_CHECK_CLOSE(r1si.x(),   2.0f, 0.1f);
	BOOST_CHECK_CLOSE(r1si.y(),  -3.0f, 0.1f);
	BOOST_CHECK_SMALL(Anglef(r1si.phi()).smallestDifferenceValue(Anglef(0.0f)), 0.1f);

}

/*
Tests:
- subscribeTransform
 */

void callbackSubscribeTransformTest(RigidTransform2f tf, Time timestamp)
{
	std::cout << "callbackSubscribeTransformTest: " << print(tf) << std::endl;
}

BOOST_AUTO_TEST_CASE(SubscribeTransformTest)
{
	Authority authority("/SubscribeTransformTest/Authority");

	Time t0 = Time::now();
	Time t1 = t0+Duration::seconds(1);
	Time t2 = t0+Duration::seconds(2);
	Time t3 = t0+Duration::seconds(3);


	RigidTransform2f A1 (1.0f, 0.0f, Degreef(-89.99f));
	RigidTransform2f B1 (3.0f, 0.0f, Degreef( 89.99f));
	RigidTransform2f C1 (1.0f, 0.0f, 0.0f);

	authority.addTransformLink("A", "R");
	//authority.addTransformLink("B", "A"); // missing here
	authority.addTransformLink("C", "B");

	authority.publishTransform("A", A1 , t1);

	authority.publishTransform("B", B1 , t1);

	authority.publishTransform("C", C1 , t1);


	// transform chain is not complete -> XTransform exception
	BOOST_CHECK_THROW(authority.subscribeTransform<RigidTransform2f>("C","R",callbackSubscribeTransformTest), TransformerBase::XTransform);


	// complete transform chain
	authority.addTransformLink("B", "A");

	authority.subscribeTransform<RigidTransform2f>("C","R",callbackSubscribeTransformTest);

	authority.publishTransform("C", C1 , t2);

	RigidTransform2f C3 (2.0f, 0.0f, 0.0f);
	authority.publishTransform("C", C3 , t3);

	MIRA_SLEEP(1000);
}

BOOST_AUTO_TEST_CASE(EigenAlignmentTests)
{
	Authority authority("/EigenAlignmentTests");

	Time t0  = Time::now();
	Time tm1 = t0-Duration::seconds(1);
	Time t1  = t0+Duration::seconds(1);

	authority.addTransformLink("A3", "R");
	
	RigidTransformCov3f tf1;
	authority.publishTransform("A3", tf1, tm1);	
	// will cause Eigen assertion if alignment is wrong
}
