/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ProcessLoader.C
 *    Loader plugin that parses the process tag
 *
 * @author Erik Einhorn
 * @date   2013/03/23
 */

#include <platform/Process.h>

#include <boost/algorithm/string.hpp>
#include <boost/tuple/tuple.hpp>

#include <vector>
#include <list>

#include <utils/Functionals.h>
#include <utils/Path.h>
#include <utils/PathFinder.h>

#include <loader/Loader.h>
#include <fw/Framework.h>
#include <fw/RemoteModule.h>

#include "private/ProcessSpawnManager.h"

namespace mira {

inline const Path& getMiraPath()
{
	static Path miraPath;
	if(miraPath.empty()) {
#ifdef MIRA_WINDOWS
# ifdef NDEBUG
		miraPath = findProjectFile("bin/mira.exe");
# else
		miraPath = findProjectFile("bin/mira_d.exe");
# endif
#else
		miraPath = findProjectFile("bin/mira");
#endif
	}
	return miraPath;
}

inline const Path& getMiraguiPath()
{
	static Path miraPath;
	if(miraPath.empty()) {
#ifdef MIRA_WINDOWS
		miraPath = findProjectFile("bin/miragui.exe");
#else
		miraPath = findProjectFile("bin/miragui");
#endif
	}
	return miraPath;
}

///////////////////////////////////////////////////////////////////////////////

/**
 * Loader plugin that allows to parse the process tag to start other
 * mira processes from within the same configuration file.
 *
 */
class ProcessLoader : public ConfigurationLoaderPlugin
{
	MIRA_OBJECT(ProcessLoader)
public:

	ProcessSpawnManager mProcessManager;

	virtual std::vector<std::string> getTags() const
	{
		return boost::assign::list_of("process")("using");
	}

	void removeComments(XMLDom::sibling_iterator& node)
	{
		XMLDom::comment_iterator comment = node.comment_begin();
		while(comment != node.comment_end())
			comment = comment.remove();
		for(XMLDom::sibling_iterator it=node.begin(); it!=node.end(); ++it)
			removeComments(it);
	}

	virtual void parseNode(const XMLDom::const_sibling_iterator& node,
	                       ConfigurationLoader* loader)
	{
		if (*node == "process")
			parseProcessNode(node,loader);
		else if (*node == "using")
			parseUsingNode(node,loader);
	}

	void parseUsingNode(const XMLDom::const_sibling_iterator& node,
	                    ConfigurationLoader* loader)
	{
		ResourceName name   = node.get_attribute<std::string>("name");
		ResourceName asName = node.get_attribute<std::string>("as", name.leaf());
		ResourceName ns     = loader->getContext()["namespace"];

		name   = ResourceName::makeFullyQualified(name, ns);
		asName = ResourceName::makeFullyQualified(asName, ns);

		mUsings.root().add_child("using").add_attribute("name",name.str())
		                                 .add_attribute("as",asName.str());
	}

	void parseProcessNode(const XMLDom::const_sibling_iterator& node,
	                      ConfigurationLoader* loader)
	{
		std::string name;
		auto nameIt = node.find_attribute("name");
		if(nameIt != node.attribute_end())
			name = nameIt.value();

		ProcessSpec* p;
		bool specAlreadyExisted = false;

		boost::tie(p, specAlreadyExisted) = getProcessSpec(name);

		auto portIt = node.find_attribute("port");
		if(portIt != node.attribute_end())
			p->port = portIt.value();

		auto connectIt = node.find_attribute("connect");
		if(connectIt != node.attribute_end())
			p->connect = (connectIt.value()=="true");

		auto guiIt = node.find_attribute("gui");
		if(guiIt != node.attribute_end())
			p->gui = (guiIt.value()=="true");

		auto argsIt = node.find_attribute("args");
		if(argsIt != node.attribute_end())
			p->extraargs += argsIt.value() + " ";

		auto respawnIt = node.find_attribute("respawn");
		if(respawnIt != node.attribute_end())
			p->respawn = (respawnIt.value()=="true");

		auto requiredIt = node.find_attribute("required");
		if(requiredIt != node.attribute_end())
			p->required = (requiredIt.value()=="true");

		auto userIt = node.find_attribute("user");
		if(userIt != node.attribute_end())
			p->user = userIt.value();

		auto machineIt = node.find_attribute("machine");
		if(machineIt != node.attribute_end()) {
			// check whether the machine attribute has changed
			if(specAlreadyExisted && p->machine!=machineIt.value()) {
				if(machineIt.value()=="thisprocess") {
					MIRA_THROW(XInvalidConfig, "A process with the same name was already declared with a different machine attribute. "
						"The special value 'thisprocess' must be specified already with the first occurrence of the declared process.");
				}
				if(!p->machine.empty()) {
					MIRA_THROW(XInvalidConfig, "A process with the same name was already declared with a different machine attribute.");
				}
			}
			p->machine=machineIt.value();
		}

		auto sshpassIt = node.find_attribute("sshpass");
		if(sshpassIt != node.attribute_end())
			p->sshpass = sshpassIt.value();

		auto saveconfigIt = node.find_attribute("save_processed_config");
		if(saveconfigIt != node.attribute_end())
			p->saveconfig = saveconfigIt.value();

		auto executableIt = node.find_attribute("executable");
		if(executableIt != node.attribute_end())
			p->executable = executableIt.value();

		auto envIt = node.find_attribute("env");
		if(envIt != node.attribute_end()) {
			p->env = Process::Environment();
			p->env->insertList(envIt.value());
		}

		// extract the content of the node, which will be the configuration
		// of the new mira process

		// copy the the process tag and all content into the config of the process
		// TODO: see #650
		//XMLDom::sibling_iterator newnode = p->config.root().add_child(node);
		XMLDom::sibling_iterator newnode = p->config.root().add_child("namespace");
		newnode.replace(node);
		newnode = "namespace"; // rename node back to "namespace"

		// remove all attributes (that come with the process tag)
		while(newnode.attribute_begin()!=newnode.attribute_end())
			newnode.remove_attribute(newnode.attribute_begin());

		// add the name of the current namespace as attribute
		newnode.add_attribute("name", loader->getContext()["namespace"]);

		// 'thisprocess' is handled as special case, which disables the
		// process tag and launches the whole content within THIS process
		if(p->machine=="thisprocess") {
			// parse the content
			loader->parse(node);
		}


	}


	virtual void beginDocument(ConfigurationLoader* ioLoader)
	{
		mProcessList.clear();
	}

	virtual void endDocument(ConfigurationLoader* ioLoader)
	{
		mProcessManager.startWatchdog(
				boost::bind(&ProcessLoader::requiredProcessTerminated, this,_1));

		foreach(auto& p, mProcessList)
			startProcess(p.get());
	}

	void requiredProcessTerminated(const ProcessSpawnManager::ProcessInfo& info)
	{
		MIRA_LOG(NOTICE) << "Required process '" << info.getName() << "' "
		                    "was terminated. Stopping ourselves ...";
		MIRA_FW.requestTermination(-1);
	}


private:

	XMLDom mUsings;

	struct ProcessSpec : boost::noncopyable
	{
		std::string name;

		std::string port;
		std::string user;
		std::string machine;
		bool connect;
		bool gui;
		std::string extraargs;
		bool respawn;
		bool required;

		std::string sshpass;
		std::string saveconfig;

		std::string executable;
		boost::optional<Process::Environment> env;

		XMLDom config;


		ProcessSpec() {
			// the defaults for each process
			port = "1234";
			connect = true;
			gui = false;
			respawn = false;
			required = false;
		}
	};

	std::list<std::unique_ptr<ProcessSpec>> mProcessList;

private:

	// returns the ProcessSpec and a flag whether the spec existed before or not
	boost::tuple<ProcessSpec*, bool> getProcessSpec(const std::string& name)
	{
		if(!name.empty()) {
			// look for process with the given name in our list
			foreach(auto& p, mProcessList)
				if(p->name==name) { // already exists?
					return boost::make_tuple(p.get(), true);
				}
		}

		// process does not exist yet, or name is empty: create new spec
		ProcessSpec* p = new ProcessSpec;
		p->name = name;
		mProcessList.push_back(std::unique_ptr<ProcessSpec>(p));
		return boost::make_tuple(p, false);
	}

	void startProcess(ProcessSpec* p)
	{
		// get executable and arguments
		Path executable;
		std::vector<std::string> args;
		bool configNeedsQuotes=false;
		std::string childHost;
		boost::optional<Process::Environment> processEnvironment;

		childHost = p->machine;
		// THIS PROCESS
		if(p->machine=="thisprocess") {
			// 'thisprocess' is handled as special case, the content of
			// the process tag was already handled in parseProcessNode
			return;

		// LOCAL MACHINE
		} else if(p->machine.empty() || p->machine=="localhost" || p->machine=="127.0.0.1") { // local
			if(childHost.empty()) // machine param could have been empty
				childHost = "127.0.0.1";

			if (p->executable.empty()) {
				if(p->gui) executable = getMiraguiPath();
				      else executable = getMiraPath();
			}
			else
				executable = p->executable;

			if(p->env) {
				// get current process environment
				processEnvironment = Process::Environment::systemEnvironment();
				// and add/overwrite user defined variables
				processEnvironment->insert(p->env.get());
			}

		// REMOTE MACHINE
		} else {

			if(p->sshpass.empty())
				executable = "ssh";
			else {
				// use sshpass for password authentication
				executable = "sshpass";
				args.push_back("-p");
				args.push_back(p->sshpass);
				args.push_back("ssh");
			}

			// this flag is necessary to make sure, that ssh sends the SIGHUP
			// signal when itself gets the SIGINT or SIGTERM
			args.push_back("-t");

			std::string machine;
			if(!p->user.empty())
				machine = p->user + "@" + p->machine;  // user@hostname
			else
				machine = p->machine;

			args.push_back(machine);

			// build the command that is sent via ssh (including environment variables and the executable)
			std::string command;

			// environment variables
			if(p->env) {
				auto envp = p->env->envp();
				foreach(auto e, envp)
					command+= e+" ";
			}

			if (p->executable.empty()) {
				if(p->gui) command += "miragui";
				      else command += "mira";
			} else
				command += p->executable;

			args.push_back(command);

			// when using ssh the shell is invoked which
			// needs quotes around our long config argument :(
			configNeedsQuotes = true;
		}


		// if connect==true, launch child with -p<port> parameter and add it
		// to our known framworks
		if(p->connect) {
			args.push_back("-p" + p->port); // add -p<port> parameter for our child
			// ... and add the child as known framework
			if(MIRA_FW.getRemoteModule())
				MIRA_FW.getRemoteModule()->addKnownFramework(childHost+":"+p->port);
		}

		// extract specified additional parameters
		boost::trim(p->extraargs);
		std::vector<std::string> addargs;
		if(!p->extraargs.empty()) {
			boost::split(addargs, p->extraargs, boost::algorithm::is_any_of(" \t\n"),
						 boost::algorithm::token_compress_on);
		}
		std::copy(addargs.begin(),addargs.end(),std::back_inserter(args));

		// and add our config string as parameter
		args.push_back("--config-string");

		// now prepare the config xml ...
		// add all usings that we came across
		auto first = p->config.root().begin();
		for(auto it=mUsings.root().begin(); it!=mUsings.root().end(); ++it)
			first.insert_before(it);

		// create string from config xml
		std::string config = p->config.saveToString();

		if(!p->saveconfig.empty()) // optional file for debugging
			p->config.saveToFile(p->saveconfig);

		// add single quotes to make ssh happy
		if(configNeedsQuotes)
		{
			// escape quotes in config to avoid breaking the ssh command
			boost::replace_all(config, "'", "'\\\''");
			args.push_back("'"+config+"'");
		}
		else
			args.push_back(config);

		mProcessManager.startProcess(executable, args, p->name, p->respawn, p->required, processEnvironment);
	}

};

///////////////////////////////////////////////////////////////////////////////
}

MIRA_CLASS_REGISTER(mira::ProcessLoader, mira::ConfigurationLoaderPlugin)

