/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file LinkLoader.C
 *    Implementation of a LinkLoader.
 *
 * @author Tim Langner
 * @date   2010/10/11
 */

#include <serialization/XMLSerializer.h>
#include <transform/RigidTransform.h>

#include <fw/Framework.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

template <typename T>
bool staticTransform(const XMLDom::const_iterator& node, const std::string& nodeID)
{
	try
	{
		XMLDeserializer deserializer(node);
		T transform;
		deserializer.deserializeFromNode("link", transform);
		// publish the transform that the user has specified.
		MIRA_FW.getTransformer()->publishTransform(nodeID, transform, Time::now());
	}
	catch(XMLDeserializer::XMemberNotFound&)
	{ // catch MemberNotFound exception only and show other errors to the user
		return false;
	}
	return true;
}

///////////////////////////////////////////////////////////////////////////////

/**
 * Loader plugin used to create transformation links between frames.
 */
class LinkLoader : public ConfigurationLoaderPlugin
{
	MIRA_OBJECT(LinkLoader)
public:
	virtual std::vector<std::string> getTags() const {
		return boost::assign::list_of("link");
	}

	virtual void parseNode(const XMLDom::const_iterator& node,
	                       ConfigurationLoader* loader);
};

void LinkLoader::parseNode(const XMLDom::const_iterator& node,
                           ConfigurationLoader* loader)
{
	auto iParent = node.find_attribute("parent");
	auto iChild  = node.find_attribute("child");
	if (iParent == node.attribute_end() ||
		iChild  == node.attribute_end())
		MIRA_THROW(XInvalidConfig, "Error in <link>. "
		           "No parent or child attribute specified. "
		           "Usage <link parent=\"ParentFrame\" child=\"ChildFrame\" />.");

	std::string  parent = iParent.value();
	std::string  child  = iChild.value();
	ResourceName ns     = loader->getContext()["namespace"];

	std::string parentID = MIRA_FW.getNameRegistry().resolve(parent, ns);
	std::string childID  = MIRA_FW.getNameRegistry().resolve(child, ns);

	MIRA_FW.getTransformer()->addLink(childID, parentID);

	if ( node.begin() != node.end() || node.has_attribute("X"))
	{
		if (!staticTransform<RigidTransformCov3f>(node, childID) &&
			!staticTransform<RigidTransform3f>(node, childID) &&
			!staticTransform<RigidTransformCov2f>(node, childID) &&
			!staticTransform<RigidTransform2f>(node, childID) )
			MIRA_THROW(XInvalidConfig, "Error in <link>. "
			           "Transformation given from parent='" << parentID << 
			           "' to child='" << childID <<"' is not valid.");
	}
}

///////////////////////////////////////////////////////////////////////////////

}

MIRA_CLASS_REGISTER(mira::LinkLoader, mira::ConfigurationLoaderPlugin)
