/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TapeRecorder.h
 *    Class for recording data to tape.
 *
 * @author Tim Langner
 * @date   2010/12/28
 */

#ifndef _MIRA_TAPERECORDER_H_
#define _MIRA_TAPERECORDER_H_

#include <serialization/Serialization.h>
#include <fw/Framework.h>
#include <fw/Tape.h>
#include <fw/PerformanceStatistics.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Contains all settings for a recorded channel like compression, codecs,...
 */
class RecordedChannelInfo
{
public:

	RecordedChannelInfo() :
		compress(false),
		interval(Duration::microseconds(0)),
		avoidDataLoss(false) {}

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("Codecs", codecs, "The codecs used for this channel",
		         std::list<BinarySerializerCodecPtr>());
		r.member("Compress", compress, "Should the data be stored compressed", false);
		r.member("Interval", interval, "Stores only data in that interval (if 0 stores all data)",
		         Duration::microseconds(0));
		r.member("AvoidDataLoss", avoidDataLoss,
		         "Avoid data loss while recording this channel by reading all data from a channel since last recording",
		         false);
	}

	std::list<BinarySerializerCodecPtr> codecs;
	bool compress;
	Duration interval;
	bool avoidDataLoss;
};

/// A map of recorded channel settings
typedef std::map<std::string, RecordedChannelInfo> RecordedChannelInfoMap;

/**
 * A config that can be passed to the TapeRecorder which contains all settings
 * that should be used for recording
 */
class TapeRecorderConfig
{
public:

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("Channels", channels, "The channels to be recorded");
		r.member("MaxMessageBlockSize", maxMessageBlockSize,
		         "The maximum size of a message block in bytes",
		         64*1024*1024); // default size is 64MB
		r.member("Filename", filename, "The filename for the output tape");
	}

	RecordedChannelInfoMap channels;
	uint32 maxMessageBlockSize;
	Path filename;
};
/**
 * @ingroup FWModule
 * Class for recording channel data to tape.
 */
class MIRA_FRAMEWORK_EXPORT TapeRecorder
{
public:

	/// Informations about a recorded channel
	class ChannelInfo : public RecordedChannelInfo
	{
	public:
		ChannelInfo() :
			recordCounter(0) {}
		Time expected;
		uint64 recordCounter;
		TypeMetaPtr meta;
		MetaTypeDatabase metaDB;
		PerformanceStatistics stats;
	};

	/// Typedef for ChannelInfo pointers
	typedef std::map<std::string, ChannelInfo> ChannelMap;

public:
	TapeRecorder();

	virtual ~TapeRecorder();

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		r.property("MaxMessageBlockSize", mMaxMessageBlockSize,
		           "The maximum size of a message block in bytes");
	}

	/**
	 * Add a channel to the list of recorded channels.
	 * @param[in] channelID Name of the channel
	 * @param[in] codecs A list of used codecs or empty if none
	 * @param[in] compress Should the channel data be recorded in a compressed
	 *            format (zip)
	 * @param[in] interval Interval in which the data gets recorded.
	 *            E.g. if a channel changes every 10 ms and the interval is 100 ms,
	 *            only 1 out of 10 changes are recorded.
	 * @param[in] avoidDataLoss Avoid data loss on the channel during recording
	 *            by using subscribeInterval() instead of subscribe()
	 */
	void addChannel(const std::string& channelID, 
	                const std::list<BinarySerializerCodecPtr>& codecs = std::list<BinarySerializerCodecPtr>(),
	                bool compress = false, Duration interval = Duration::microseconds(0), bool avoidDataLoss = false);

	/**
	 * Add a channel to the list of recorded channels by providing an ID
	 * and an info object that contains the settings for the channel.
	 * @param[in] channelID Name of the channel
	 * @param[in] info Settings for recording the channel
	 */
	void addChannel(const std::string& channelID,
	                const RecordedChannelInfo& info);

	/**
	 * Start recording to file.
	 * @param[in] file Filename of the tape
	 */
	void record(const Path& file);

	/**
	 * Start recording to file using a recorder config that
	 * contains channels and settings for recording.
	 * @param[in] config A config containing all settings for recording
	 */
	void record(const TapeRecorderConfig& config);

	/**
	 * Stop recording
	 */
	void stop();

	/**
	 * Set the maximum size of the chunks
	 * @param[in] maxMessageBlockSize Size of a chunk in bytes
	 */
	void setMaxMessageBlockSize(uint32 maxMessageBlockSize)
	{
		mMaxMessageBlockSize = maxMessageBlockSize;
	}

	/**
	 * Returns whether TapeRecorder is currently gathering data after record was called.
	 */
	bool isRecording() const
	{
		return mIsRecording;
	}

	const PerformanceStatistics& getStats(const std::string& name);

protected:

	// normal callback for data changes (writes data to tape)
	void valueChanged(ChannelRead<void> value, bool forceStorage = false);
	void intervalValueChanged(ChannelReadInterval<void> values, bool forceStorage = false);

private:

	struct MessageToStore
	{
		StampedHeader header;
		std::string channelID;
		std::string typeName;
		Time enqueueTime;
		Buffer<uint8> buf;
		ChannelInfo* info;
	};

	void storageThread();
	void storeMessage(MessageToStore& msg);

protected:

	Authority mAuthority;
	uint32 mMaxMessageBlockSize;
	Tape mTape;
	bool mIsRecording;
	bool mFirstCallbackMessage;
	ChannelMap mChannels;

	boost::thread mStorageThread;
	boost::condition_variable mStorageThreadCondition;
	boost::mutex mStorageMutex;

	std::deque<MessageToStore> mStorageQueue;
	volatile bool mStorageThreadTerminationRequest;
};

///////////////////////////////////////////////////////////////////////////////

}

#endif
