/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Tape.h
 *    A Tape is a representation of recorded channel data in a binary file.
 *
 * @author Tim Langner
 * @date   2010/12/27
 */

#ifndef _MIRA_TAPE_H_
#define _MIRA_TAPE_H_

#ifndef Q_MOC_RUN
#include <boost/optional.hpp>
#endif

#include <utils/Path.h>
#include <fw/Channel.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup FWModule
 * A tape is a binary file that contains recorded/serialized data of one
 * or multiple channels. The tape file can be played back later to review and
 * analyze the recorded data e.g. for testing algorithms.
 * The format of a tape is as follows: (numbers in () list the size in bytes)
 * <FileHeader>
 * 	-version (4)
 *  -time of recording in nanoseconds since 00:00:00 01.01.1970 (8)
 *  -time difference of recording machine to UTC time in nanoseconds (8)
 *  -number of message blocks (4)
 *  -offset to first channel info field (8)
 *
 *  Every other field has a 5 byte header:
 * <FieldHeader>
 *  -type of field (1)
 *  -size (4)
 *
 * <MessageBlockField>
 *  -number of messages (4)
 *  -block size (4)
 *  -time of first message in nanoseconds, relative to time of recording (8)
 *  -time of last message in nanoseconds, relative to time of recording (8)
 *
 * <ChannelInfoField>
 *  -offset to next channel info field (8)
 *  -time of first message in nanoseconds, relative to time of recording (8)
 *  -time of last message in nanoseconds, relative to time of recording (8)
 *  -name (size (4) + size bytes)
 *  -type (size (4) + size bytes)
 *  -meta data (size (8) + size bytes)
 *  -offset to index table (8)
 *
 * <MessageField>
 *  -time of message in nanoseconds, relative to time of recording (8)
 *  -name (size (4) + size bytes)
 *  -frameID (size(4) + size bytes)
 *  -sequenceID (4)
 *  -size of (compressed) data (4)
 *  -compression flag tells us if message is compressed(1)
 *  -if compression flag is set size of uncompressed data (4)
 *  -data (size of (compressed) data bytes)
 *
 * <IndexField>
 *  -offset to block field (8)
 *  -relative offset in message block to message (8)
 *  -time of message in nanoseconds, relative to time of recording (8)
 * 
 * History of Tape versions:
 * - 5.0 (0x00050000) current
 *   - message header now contains frame id and sequence id of the data
 *   - the real message data now only contains the serialized value
 * - 4.0 (0x00040000)
 *   - added time zone offset to file header
 * - 3.0 (0x00030000)
 *   - added meta data to channel informations
 * - 2.0 (0x00020000)
 *   - using 64bit instead of 32bit for all sizes and offsets in tape files
 * - 1.1 (0x00010001)
 *   - using nanosecond instead of microsecond resolution for all time offsets 
 *     and values in tapes
 * - 1.0 (0x00010000)
 *   - initial version
 */
class MIRA_FRAMEWORK_EXPORT Tape
{
public:
	/// size of the header packet
	static const uint32 sHeaderSize;
	// size of a message block packet
	static const uint32 sMessageBlockSize;

	/// The open mode for a tape
	enum OpenMode
	{
		READ,
		WRITE,
		INFO,	/// opens tape for reading information about channels and only
				/// (does not read complete message block list)
				/// can be used for fast opening and displaying informations
				/// e.g in a tape open file dialog
	};

	/// The type of the header specifying the following packet
	enum HeaderType
	{
		MESSAGEBLOCK = 0x0A,
		CHANNELINFO = 0x0B,
		MESSAGE = 0x0C,
		INDEX = 0x0D
	};

	/// Header containing type and size of the following packet
	struct Header
	{
		char type;		///< Type of the packet (see HeaderType)
		uint32 size;	///< Size of the packet
	};

	/// Index entry for a message in the tape
	struct MessageIndex
	{
		uint64 block;			///< Offset in bytes of the containing block in the tape
		uint64 offset;			///< Offset in bytes of the message in the containing block
		Duration timeOffset;	///< Time offset of the message relative to start time of tape
	};

	/// Information about a channel in a tape
	struct ChannelInfo
	{
		typedef std::vector<MessageIndex> MessageVector;
		uint64 offset;				///< Offset in bytes of the info packet in the tape
		uint64 offsetToIndexTable;	///< Offset in bytes of the index table for the channel
		Duration firstMessageOffset;///< Time offset of the first message in the channel
		Duration lastMessageOffset;	///< Time offset of the last message in the channel
		std::string name;			///< Name of the channel
		std::string type;			///< Typename of the channel
		TypeMetaPtr meta;			///< Type meta information
		MetaTypeDatabase metaDB;	///< Database of meta information needed by this type
		MessageVector messages;		///< Messages
		uint64 dataSize;			///< Size of all messages in bytes
	};

	/// Information about a tape file
	struct FileInfo
	{
		std::fstream file;			///< The file stream
		Time start;					///< Start recording time
		Duration timezoneOffset;	///< Offset of the timezone on the recording machine to UTC
		std::string filename;		///< Name of the tape
		uint32 version;				///< Tape version
		uint32 nrBlocks;			///< Nr of message data blocks
		uint32 nrChannels;			///< Nr of channels
		uint64 offsetToFirstInfo;	///< Offset to the first channel info packet
	};

	/// Struct for a message block in a tape
	struct MessageBlock
	{
		uint64 offset;					///< Offset in bytes of the block in the tape
		uint32 nrMessages;				///< Nr of contained messages
		uint32 size;					///< Size of the block
		Duration firstMessageOffset;	///< Time offset of the first message in the block
		Duration lastMessageOffset;		///< Time offset of the last message in the block
		Buffer<uint8> buffer;			///< Buffer for the block data
	};

	/// Struct for message data in a tape
	struct Message
	{
		bool compressed;		///< Is message compressed
		std::string name;		///< Name of the messages channel
		std::string frameID;	///< Frame id of the message
		uint32 sequenceID;		///< Sequence id of the message
		Buffer<uint8> data;		///< Buffer for the message data
		uint32 uncompressedSize;///< Uncompressed size of the message
	};

	/// maps channel information to a channel names
	typedef std::map<std::string, ChannelInfo> ChannelMap;
	/// maps a message block to an offset
	typedef std::map<uint64, MessageBlock> MessageBlockMap;
	/// maps a message to an time offset
	typedef std::multimap<Duration, Message> MessageMap;

	Tape() :
		mIsOpen(false),
		mWaitForAlteredStartTime(false),
		mMaxMessageBlockSize(64 * 1024* 1024),
		mSortWindowSize(Duration::seconds(2))
	{}

	~Tape();

	/**
	 * Open the tape for reading or writing.
	 * @param[in] file The filename of the tape
	 * @param[in] mode The opening mode (read or write)
	 */
	void open(const Path& file, OpenMode mode);

	/**
	 * Close the tape
	 */
	void close();

	/**
	 * Try to repair the tape given by file and store repaired version in
	 * outFile.
	 * @param[in] file The filename of the tape to be repaired
	 * @param[in] outFile The filename of the repaired tape
	 */
	void repair(const Path& file, const Path& outFile);

	/**
	 * If this method is called in write mode 
	 * (must be called before writing the first message to the tape)
	 * all recorded data is written to file after setting the start time of the
	 * tape by calling alterStartTime(). Time offsets of all message written
	 * until alterStartTime() is called will be corrected relative to the new
	 * start time. This can be used to write "old" data to the tape first and
	 * specify the start time of recording later.
	 * @throw XIO if the tape is not opened in write mode
	 * @throw XInvalidConfig if data was already written to the tape.
	 */
	void waitForAlteredStartTime();

	/**
	 * Alters the start time of a tape in write mode.
	 * Can only be called if waitForAlteredStartTime() was called before
	 * or no data has been written to the tape yet.
	 * @throw XIO if the tape is not opened in write mode
	 * @throw XInvalidConfig if waitForAlteredStartTime() was not called before
	 *        or data was already written to the tape.
	 */
	void alterStartTime(const Time& startTime);

	/**
	 * Write the content of a channel into the tape
	 * @throw XIO if the tape is not opened in write mode
	 * @param value The channel that data gets serialized and written to tape
	 * @param compress If true data is compressed using zip compression
	 * @param meta meta information about the type of the data
	 * @param meta database that contains meta information about the types used
	 *        by the data type itself (e.g. class members)
	 * @return Returns the number of bytes of data, added to the tape
	 */
	template <typename T>
	std::size_t write(ChannelRead<T> value, bool compress = false,
	                  TypeMetaPtr meta = TypeMetaPtr(),
	                  const MetaTypeDatabase& metaDB = MetaTypeDatabase())
	{
		if (!mIsOpen || mMode != WRITE)
			MIRA_THROW(XIO, "Tape is not opened in write mode");
		return write(value.getChannelID(), value.getTypename(),
		             value->timestamp, value.readSerialized(),
		             compress, meta, metaDB);
	}

	/**
	 * Write the content of a channel into the tape
	 * @throw XIO if the tape is not opened in write mode
	 * @param value The channel that data gets serialized and written to tape
	 * @param codecs A list of codecs. If this list contains a codec that
	 *        matches the type of the channel it is used to encode its value.
	 * @param compress If true data is compressed using zip compression
	 * @param meta meta information about the type of the data
	 * @param meta database that contains meta information about the types used
	 *        by the data type itself (e.g. class members)
	 * @return Returns the number of bytes of data, added to the tape
	 */
	template <typename T>
	std::size_t write(ChannelRead<T> value,
	                  std::list<BinarySerializerCodecPtr>& codecs,
	                  bool compress = false, TypeMetaPtr meta = TypeMetaPtr(),
	                  const MetaTypeDatabase& metaDB = MetaTypeDatabase())
	{
		if (!mIsOpen || mMode != WRITE)
			MIRA_THROW(XIO, "Tape is not opened in write mode");
		return write(value.getChannelID(), value.getTypename(),
		             value->timestamp, value.readSerialized(codecs),
		             compress, meta, metaDB);
	}

	/**
	 * Write the stamped data into the tape
	 * @throw XIO if the tape is not opened in write mode
	 * @param channelID The name of the channel
	 * @param value The value that contains the stamped header information
	 *        and the data that gets serialized and written to tape
	 * @param compress If true data is compressed using zip compression
	 * @param meta meta information about the type of the data
	 * @param meta database that contains meta information about the types used
	 *        by the data type itself (e.g. class members)
	 * @return Returns the number of bytes of data, added to the tape
	 */
	template <typename T>
	std::size_t write(const std::string& channelID, const Stamped<T>& value,
	                  bool compress = false, TypeMetaPtr meta = TypeMetaPtr(),
	                  const MetaTypeDatabase& metaDB = MetaTypeDatabase())
	{
		Buffer<uint8> buffer;
		BinaryBufferSerializer s(&buffer);
		s.serialize(value.value(),false);
		return write(channelID, typeName<T>(), value.timestamp,
		             value.frameID, value.sequenceID,
		             buffer, compress, meta, metaDB);
	}

	/**
	 * Write the serialized data into the tape. The stamped information is passed
	 * via the stamped header parameter.
	 * @throw XIO if the tape is not opened in write mode
	 * @param channelID The name of the channel
	 * @param typeName The typename of the data
	 * @param header The stamped information for the data
	 * @param data The serialized data that gets written to tape
	 * @param compress If true data is compressed using zip compression
	 * @param meta meta information about the type of the data
	 * @param metaDB database that contains meta information about the types used
	 *        by the data type itself (e.g. class members)
	 * @return Returns the number of bytes of data, added to the tape
	 */
	std::size_t write(const std::string& channelID, const std::string& typeName,
	                  const StampedHeader& header,
	                  const Buffer<uint8>& data, bool compress = false,
	                  TypeMetaPtr meta = TypeMetaPtr(),
	                  const MetaTypeDatabase& metaDB = MetaTypeDatabase())
	{
		return write(channelID, typeName, header.timestamp, header.frameID,
		             header.sequenceID, data, compress, meta, metaDB);
	}


	/**
	 * Low level write method to write serialized binary data to the tape.
	 * The channelID, the typename of the data, the time of the message,
	 * the frameID, the sequenceID and the actual binary data must be specified.
	 * Additionally a flag can be specified if the data should be compressed
	 * @return Returns the number of bytes of data, added to the tape
	 */
	std::size_t write(const std::string& channelID, const std::string& typeName,
	                  const Time& time, const std::string& frameID, uint32 sequenceID,
	                  const Buffer<uint8>& data, bool compress = false,
	                  TypeMetaPtr meta = TypeMetaPtr(),
	                  const MetaTypeDatabase& metaDB = MetaTypeDatabase());
	/**
	 * Set the maximum size of the chunks
	 * @param[in] size Size of a chunk in bytes
	 */
	void setMaxMessageBlockSize(uint32 size)
	{
		mMaxMessageBlockSize = size;
	}

	/**
	 * Messages get sorted before they are written to a tape to guarantee
	 * a correct temporal order. Messages are collected in a sorted temporary
	 * list. While new messages arrive older messages get written to file when
	 * they move out of the sort window. One can specify the size of the sorting
	 * window by calling this function.
	 */
	void setSortWindowSize(const Duration& interval)
	{
		mSortWindowSize = interval;
	}

	/**
	 * Set an alternative start time for recording.
	 * This is used when creating new tapes out of existing ones to use their
	 * start time instead of now.
	 * @param[in] time Start time of recording
	 */
	void setStartTime(Time time)
	{
		mFile.start = time;
		mFile.timezoneOffset = mFile.start.toLocal() - mFile.start;
	}

	/**
	 * Get the time when the recording started.
	 * @throw XIO if the tape is not opened
	 * @return The time when recording was started.
	 */
	Time getStartTime() const
	{
		if (!mIsOpen)
			MIRA_THROW(XIO, "Tape is not opened.");
		return mFile.start;
	}

	/**
	 * Get the time when the recording started in the
	 * local timezone of the machine the recording took place (Time at place of recording).
	 * @throw XIO if the tape is not opened
	 */
	Time getLocalStartTime() const
	{
		if (!mIsOpen)
			MIRA_THROW(XIO, "Tape is not opened.");
		return mFile.start + mFile.timezoneOffset;
	}

	/**
	 * Get the time of the last entry.
	 * @throw XIO if the tape is not opened
	 * @return The time of the last entry in the tape
	 */
	Time getEndTime() const
	{
		if (!mIsOpen)
			MIRA_THROW(XIO, "Tape is not opened.");
		if (mMessageBlocks.size()==0)
			return Time::unixEpoch();
		return mFile.start + mMessageBlocks.rbegin()->second.lastMessageOffset;
	}

	/**
	 * Get a list of channels in the tape
	 * @throw XIO if the tape is not opened
	 * @return A list of all channels in the tape
	 */
	const Tape::ChannelMap& getChannels() const
	{
		if (!mIsOpen)
			MIRA_THROW(XIO, "Tape is not opened.");
		return mChannels;
	}

	/**
	 * Get a list of the tapes message blocks
	 * @throw XIO if the tape is not opened
	 * @return A map containing information about each chunk in the tape.
	 */
	const Tape::MessageBlockMap& getMessageBlocks() const
	{
		if (!mIsOpen)
			MIRA_THROW(XIO, "Tape is not opened.");
		return mMessageBlocks;
	}

	/**
	 * Get the version of the tape.
	 * @throw XIO if the tape is not opened
	 * @return The version of the tape
	 */
	uint32 getVersion() const
	{
		if (!mIsOpen)
			MIRA_THROW(XIO, "Tape is not opened.");
		return mFile.version;
	}

	/**
	 * Returns the tape version that is supported by the local implementation.
	 * Note, that this is the version that is used to generate new tape files.
	 * In contrast the getVersion() method returns the version of the opened
	 * tape file that may differ from that of the local implementation.
	 */
	static uint32 getCurrentVersion();

	/**
	 * Read a message from tape at the given index.
	 * An index can be obtained from the ChannelMaps messages member
	 * @throw XIO if the tape is not opened in read mode
	 * @param index Read message at that index
	 * @param frameID Frame id of the message
	 * @param sequenceID Sequence id of the message
	 * @param[out] oData Buffer to the data.
	 *             Is resized and filled with the serialized data
	 *             from the tape at index position
	 * @param[out] oTime The time offset to the start of recording
	 *             when that message was recorded.
	 */
	void readMessage(const MessageIndex& index, std::string& frameID,
	                 uint32& sequenceID, Buffer<uint8>& oData, Duration& oTime);

	/**
	 * Same as above but allows to obtain additional information whether
	 * the data was compressed within the tape or not.
	 * @throw XIO if the tape is not opened in read mode
	 */
	void readMessage(const MessageIndex& index, std::string& frameID,
	                 uint32& sequenceID, Buffer<uint8>& oData, Duration& oTime,
	                 bool& oCompressed);


protected:

	uint64 getFileHeaderSize() const;
	void openInfo(const Path& file);
	void openRead(const Path& file);
	void closeRead();
	void openWrite(const Path& file);
	void closeWrite();
	void openRepair(const Path& file);

	void write(Duration time, const Message& message);

	void readFileHeader();
	void readChannelInfo();
	void readMessageBlocks();
	void readMessageBlock(uint64 offset);
	void writeFileHeader();
	Tape::Header readHeader();
	void writeHeader(HeaderType type, uint32 size);
	void writeChannelInfo(ChannelInfo& info);
	void startMessageBlock(Duration time);
	void finishMessageBlock();

protected:

	bool mIsOpen;
	bool mWaitForAlteredStartTime;
	OpenMode mMode;
	ChannelMap mChannels;
	std::set<std::string> mWrittenChannels;
	boost::optional<MessageBlock> mCurrentMessageBlock;
	boost::mutex mMutex;
	boost::mutex mMessageMutex;
	MessageBlockMap mMessageBlocks; ///< Pairs of offset - message block
	FileInfo mFile;                 ///< File informations
	uint64 mLastInfo;               ///< Offset to the last written info field
	uint32 mMaxMessageBlockSize;    ///< Maximum size of a message block

	MessageMap mMessages;           ///< Messages we have not yet written to file (in the sort window)
	Duration mSortWindowSize;       ///< Size of the window used for sorting messages in a tape
};

///////////////////////////////////////////////////////////////////////////////

}

#endif
