/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ChannelManager.h
 *    Manager class for all framework channels.
 *
 * @author Erik Einhorn, Tim Langner
 * @date   2010/09/06
 */

#ifndef _MIRA_CHANNELMANAGER_H_
#define _MIRA_CHANNELMANAGER_H_

#include <platform/Types.h>
#include <error/Exception.h>
#include <thread/Thread.h>

#include <fw/Channel.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * An exception that occurred whenever a channel does not exist
 */
class XUnknownChannel : public XRuntime
{
public:
	XUnknownChannel(std::string msg, const char* file=NULL, int line=0) throw() :
		XRuntime(msg, file, line) {}
};

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup FWModule
 * This manager provides functionality to create, obtain and manage
 * the channels in the framework.
 * Channels provide a time ordered queue (history) of data slots 
 * (timestamp, value pairs).
 */
class MIRA_FRAMEWORK_EXPORT ChannelManager
{
public:
	/// Informations about a channel that also provides ordering of channels by id
	struct ChannelInfo
	{
		ChannelInfo() {}

		ChannelInfo(const std::string& iID, bool iInternal = false) :
			id(iID),
			internal(iInternal) {}

		bool operator<(const ChannelInfo& other) const
		{
			return id < other.id;
		}

		bool operator==(const ChannelInfo& other) const
		{
			return id == other.id;
		}

		std::string id;
		bool internal;
	};

	ChannelManager();
	~ChannelManager();

	/// Reflect method for serialization
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.interface("IChannelManager");
		r.method("writeChannel", &ChannelManager::writeChannel, this,
		         "Writes a json representation to a channel of type T");
		r.method("readChannel", &ChannelManager::readChannel, this,
		         "Reads the latest channel data as json representation");
		r.method("getChannelList", &ChannelManager::getChannelList, this,
		         "get list of channels");
	}

public:
	/**
	 * Publishes a channel with the given id.
	 * If the channel does not exist it will be created.
	 * If the channel is a polymorphic channel (i.e. T is a pointer type of
	 * a polymorphic type) then publishing the channel will fixate the channels
	 * type and future publish() calls with different types will result in
	 * a XBadCast exception.
	 * @throw XBadCast if the channel exists but the types do not match.
	 * @param[in] channelID The channel id
	 * @param[in] publisherID The id of the publisher.
	 *            If empty the channel will not be visible in a call to
	 *            getPublishedChannels().
	 * @param[in] internal The internal flag specifies if the publisher counts not
	 *            as a public publisher to that channel (e.g. remote publishers)
	 * @param[in] type The typename of the channels type
	 * @return A concrete Channel object that the caller can convert to a
	 *         channel by providing access flags
	 */
	template<typename T>
	ConcreteChannel<T>* publish(const std::string& channelID,
	                            const std::string& publisherID,
	                            bool internal, const Typename& type)
	{
		// If exists && id and T are not the same throw Exception
		ConcreteChannel<T>* channel = obtainConcreteChannel<T>(channelID);
		channel->fixateType(); // fixate the type
		if ( !channel->isTyped() )
			channel->setTypename(type);
		{
			insertPublishedChannel(channelID, ChannelInfo(publisherID, internal));
			MIRA_LOG(DEBUG) << "'" << publisherID << "' published channel '" << channelID << "'";
		}

		if (!internal)
			remotePublishChannel(channelID, type);

		return channel;
	}

	/**
	 * Subscribes to a channel with the given id.
	 * The channel is created if not exists.
	 * @throw XBadCast if the channel exists but the types do not match.
	 * @param[in] channelID The id of the data channel to subscribe to
	 * @param[in] subscriberID The id of the subscriber.
	 *            If empty the channel will not be visible in a call to
	 *            getSubscribedChannels().
	 * @param[in] storageDuration The requested duration of how long data
	 *            is stored in the channel (history). Note that the real
	 *            history length may differ from this request because
	 *            the number of slots in the channel is limited by its maximum length.
	 *            Assuming we have only one writer to the channel
	 *            that updates the channel each T milliseconds the real storage
	 *            duration D can be computed by D = (maxSlots-1) * T.
	 * @param[in] internal The internal flag specifies if the subscriber counts not
	 *            as a public subscriber to that channel (e.g. remote subscribers)
	 * @return A concrete Channel object that the caller can convert
	 *         to a channel by providing access flags
	 */
	template<typename T>
	ConcreteChannel<T>* subscribe(const std::string& channelID,
	                              const std::string& subscriberID,
	                              const Duration& storageDuration, bool internal)
	{
		ConcreteChannel<T>* channel = obtainConcreteChannel<T>(channelID);

		assert(channel->getBuffer()!=NULL);
		channel->getBuffer()->setStorageDuration(storageDuration);
		insertSubscribedChannel(channelID, ChannelInfo(subscriberID, internal));
		MIRA_LOG(DEBUG) << "'" << subscriberID << "' subscribed to channel '" << channelID << "'";

		if (!internal)
			remoteSubscribeChannel(channelID);

		return channel;
	}

	/**
	 * Remove a publisher from the publisher lists of all channels it has published.
	 * @param[in] publisherID The id of the publisher.
	 * @param[in] internal Is the publisher an internal one that does not count
	 *            as a global publisher (only local in this framework)
	 */
	void unpublish(const std::string& publisherID, bool internal);

	/**
	 * Remove a publisher from the list of publishers of the given channel
	 * @param[in] channelID The channel ID
	 * @param[in] publisherID The id of the publisher.
	 * @param[in] internal Is the publisher an internal one that does not count
	 *            as a global publisher (only local in this framework)
	 */
	void unpublish(const std::string& channelID, const std::string& publisherID, bool internal);

	/**
	 * Remove a subscriber from the subscriber lists of all channels it
	 * is subscribed on.
	 */
	void unsubscribe(const std::string& subscriberID);

	/**
	 * Remove a subscriber from the subscriber list of channel channelID.
	 * @param[in] channelID The channel ID
	 * @param[in] subscriberID The id of the subscriber.
	 */
	void unsubscribe(const std::string& channelID, const std::string& subscriberID);

	/**
	 * Returns the existing channel with the given id.
	 * @param[in] channelID The channel ID
	 * @return Pointer to the channel
	 * @throw XBadCast if the channel exists but the types do not match.
	 * @throw XUnknownChannel if the channel does not exist.
	 */
	template<typename T>
	ConcreteChannel<T>* getConcreteChannel(const std::string& channelID)
	{
		return channel_cast<T>(getAbstractChannel(channelID));
	}

public:

	/**
	 * Get a list of all channels managed by this manager.
	 */
	std::set<std::string> getChannels() const;

	/**
	 * Get a list of all channels managed by this manager, can be filtered for
	 * only those that have subscribers and/or publishers (any combination is possible).
	 *
	 * @param[in] publishedOnly Only return channels that have (local) publishers.
	 *                          If false, this will not exclude any channels.
	 * @param[in] subscribedOnly Only return channels that have (local) subscribers
	 *                           (excluding internal subscribers). If false,
	 *                           this will not exclude any channels.
	 * @return List of channels matching the requested criteria.
	 */
	std::set<std::string> getChannelList(bool publishedOnly = false,
	                                     bool subscribedOnly = false) const;

	/**
	 * Get a list of all channels we have subscriptions on.
	 * @param[in] includeInternal If set to false (default) the returned list
	 *            excludes channels that have only internal subscribers
	 * @return List of subscribed channels
	 */
	std::set<std::string> getSubscribedChannels(bool includeInternal = false) const;

	/**
	 * Get a list of all channels that the given subscriber has subscribed on
	 * @return List of subscribed channels
	 */
	std::set<std::string> getSubscribedChannelsBy(const std::string& subscriberID) const;

	/**
	 * Returns true if the subscriber is subscribed to the given channel
	 */
	bool isSubscribedOn(const std::string& subscriberID,
	                    const std::string& channelID) const;

	/**
	 * Get a list of all channels and their types we have publishers for.
	 * @param[in] includeInternal If set to false (default) the returned list
	 *            excludes channels that have only internal publishers
	 * @return List of published channels
	 */
	std::map<std::string, Typename> getPublishedChannels(bool includeInternal = false) const;

	/**
	 * Get a list of all channels that are currently published by the given publisher
	 * @param[in] publisherID Publisher name
	 * @return List of published channels
	 */
	std::set<std::string> getPublishedChannelsBy(const std::string& publisherID) const;

	/**
	 * Returns true if the publisher has published the given channel
	 */
	bool hasPublished(const std::string& publisherID, const std::string& channelID) const;

	/**
	 * Returns a list of all channels that match the specified type.
	 * Each list item is a pair where the first entry specifies the full
	 * qualified name of the channel and the second entry specifies the
	 * type of the channel.
	 */
	std::list<std::pair<std::string, Typename>> getChannelsOfType(const Typename& type) const;

	/**
	 * Returns a list of all channels that match the specified type.
	 * Each list item is a pair where the first entry specifies the full
	 * qualified name of the channel and the second entry specifies the
	 * type of the channel.
	 */
	template <typename T>
	std::list<std::pair<std::string, Typename>> getChannelsOfType() const {
		return getChannelsOfType(typeName<T>());
	}

	/**
	 * Returns if we have at least one subscriber (including internal ones)
	 * for the given channel
	 * @param[in] channelID The channel id
	 * @return true if we have a subscriber for this channel,
	 *         false otherwise or if channel does not exist
	 */
	bool hasSubscriber(const std::string& channelID) const;

	/**
	 * Returns if we have a publisher for a given channel.
	 * @param[in] channelID The channel id
	 * @param[in] includeInternal If true also internal publishers will count as a
	 *            publisher for that channel.
	 * @return true if channel has at least one publisher, false if not
	 */
	bool hasPublisher(const std::string& channelID, bool includeInternal) const;

	/**
	 * Get the number of publishers for that channel
	 */
	uint32 getNrPublishers(const std::string& channelID) const;

	/**
	 * Get the number of subscribers to that channel
	 */
	uint32 getNrSubscribers(const std::string& channelID) const;

	/**
	 * Returns true, if the specified channel with the fully qualified
	 * channelID exits.
	 */
	bool hasChannel(const std::string& channelID) const;

	/**
	 * Returns the type id of the specified channel with the fully qualified
	 * channelID or -1, if the channel is untyped.
	 */
	int getTypeId(const std::string& channelID) const;

	/**
	 * Sets the typename of the specified channel.
	 */
	void setTypename(const std::string& channelID, const Typename& typenam);

	/**
	 * Returns the typename of the specified channel with the fully qualified
	 * channelID
	 */
	Typename getTypename(const std::string& channelID) const;

	/**
	 * Sets the type meta information for the slots of the
	 * underlying channel buffer.
	 */
	void setTypeMeta(const std::string& channelID, TypeMetaPtr meta);

	/**
	 * Returns the type meta information of the channel.
	 */
	TypeMetaPtr getTypeMeta(const std::string& channelID) const;

	/**
	 * Returns the timestamp of the data in the slot that was last written to
	 * the channel with the fully qualified channelID or an invalid time when
	 * there was no data written at all.
	 */
	Time getLastSlotTime(const std::string& channelID) const;

	/**
	 * Returns the number of slots in the channel
	 */
	std::size_t getNrOfSlots(const std::string& channelID) const;

	/**
	 * Returns the number of times data was written to the channel
	 * with the fully qualified channelID or an invalid time.
	 */
	uint64 getNrOfDataChanges(const std::string& channelID) const;

public:

	/**
	 * Just make sure the specified channel exists (by creating it if needed),
	 * without changing any attributes.
	 */
	void ensureChannel(const std::string& channelID)
	{
		obtainConcreteChannel<void>(channelID);
	}

	/**
	 * Sets storage duration for the specified channel.
	 *
	 * @copydoc ChannelBufferBase::setStorageDuration(const Duration& storageDuration)
	 */
	void setStorageDuration(const std::string& channelID, const Duration& storageDuration);

	/**
	 * Sets desired minimum number of slots for the specified channel.
	 *
	 * @copydoc ChannelBufferBase::setMinSlots(std::size_t maxSlots)
	 */
	void setMinSlots(const std::string& channelID, std::size_t minSlots);

	/**
	 * Sets the maximum number of slots for the specified channel.
	 *
	 * @copydoc ChannelBufferBase::setMaxSlots(std::size_t maxSlots)
	 */
	void setMaxSlots(const std::string& channelID, std::size_t maxSlots);

public:

	/**
	 * Writes a json value to the channel with given id.
	 */
	void writeChannel(const std::string& channelID,
	                  const json::Value& value,
	                  const Time& time = Time::now());

	/**
	 * Reads data from the channel with given id.
	 * If channel has type T it returns the json representation of T.
	 */
	json::Value readChannel(const std::string& channelID);

private:

	/**
	 * Returns the untyped channel with the given name.
	 * @param[in] channelID The channel ID
	 * @return Pointer to the channel
	 * @throw XUnknownChannel If channel does not exist
	 */
	const AbstractChannel* getAbstractChannel(const std::string& channelID) const;

	/**
	 * Returns the untyped channel with the given name.
	 * @param[in] channelID The channel ID
	 * @return Pointer to the channel
	 * @throw XUnknownChannel If channel does not exist
	 */
	AbstractChannelPtr getAbstractChannel(const std::string& channelID);

	/**
	 * Returns the untyped channel with the given name. (does not lock mChannelsMutex)
	 * @param[in] channelID The channel ID
	 * @return Pointer to the channel
	 * @throw XUnknownChannel If channel does not exist
	 */
	AbstractChannelPtr getAbstractChannelNoLocking(const std::string& channelID);

	/**
	 * Returns the untyped channel with the given name. (does not lock mChannelsMutex)
	 * @param[in] channelID The channel ID
	 * @return Pointer to the channel
	 * @throw XUnknownChannel If channel does not exist
	 */
	const AbstractChannel* getAbstractChannelNoLocking(const std::string& channelID) const;

	/**
	 * Returns the existing channel with the given id or creates a new channel
	 * if the channel does not exist.
	 * @param[in] channelID The channel ID
	 * @return Pointer to the channel
	 * @throw XBadCast if the channel exists but the types do not match.
	 */
	template<typename T>
	ConcreteChannel<T>* obtainConcreteChannel(const std::string& channelID)
	{
		boost::mutex::scoped_lock lock(mChannelsMutex);

		ConcreteChannel<T>* channel = NULL;
		try
		{
			channel = channel_cast<T>(getAbstractChannelNoLocking(channelID));
		}
		catch (XUnknownChannel&)
		{
			// if we reach here, the channel does not exist, so create a new one
			channel = new ConcreteChannel<T>(channelID);
			mChannels.insert(ChannelMap::value_type(channelID, channel));
		}
		return channel;
	}

	/** @name Remote handling */
	//@{
	void remotePublishChannel(const std::string& channelID, const Typename& type);
	void remoteUnpublishChannel(const std::string& channelID);
	void remoteSubscribeChannel(const std::string& channelID);
	void remoteUnsubscribeChannel(const std::string& channelID);
	//@}


private:

	void insertPublishedChannel(const std::string& channelID,  const ChannelInfo& info);
	void insertSubscribedChannel(const std::string& channelID,  const ChannelInfo& info);

private:

	typedef std::map<std::string, AbstractChannelPtr > ChannelMap;
	ChannelMap mChannels; ///< maps from each existing channel to a channel info object that provides further information
	mutable boost::mutex mChannelsMutex;

	class Pimpl;
	Pimpl* p;

	mutable boost::mutex mSubscriberMutex;
	mutable boost::mutex mPublisherMutex;

};

//////////////////////////////////////////////////////////////////////////////

}

#endif
