/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

#include <boost/test/unit_test.hpp>

#include <thread/Spinlock.h>

#include <thread/Thread.h>
#include <utils/Time.h>

using namespace mira;

///////////////////////////////////////////////////////////////////////////////

uint64 wasteCPU(int c)
{
	static volatile uint64 dummy=0; // volatile avoids optimizations here
	for(int i=0; i<c*1000000; ++i)
		dummy+=i;
	return dummy;
}


Spinlock lock;
int test = 0;

void thread1()
{
	MIRA_SLEEP(100);
	std::cout << "thread1 lock" << std::endl;
	lock.lock();
	std::cout << "thread1 locked" << std::endl;

	wasteCPU(100);

	test = 1;
	lock.unlock();

	wasteCPU(100);

	std::cout << "thread1 unlocked" << std::endl;
}

void thread2()
{
	MIRA_SLEEP(200);
	std::cout << "thread2 lock" << std::endl;
	lock.lock();
	std::cout << "thread2 locked" << std::endl;
	BOOST_CHECK_EQUAL(test,1);

	wasteCPU(100);

	lock.unlock();
	std::cout << "thread2 unlocked" << std::endl;
}

///////////////////////////////////////////////////////////////////////////////

BOOST_AUTO_TEST_CASE( SpinlockTest )
{
	boost::thread t1(boost::bind(&thread1));
	boost::thread t2(boost::bind(&thread2));

	t1.join();
	t2.join();
}

///////////////////////////////////////////////////////////////////////////////

BOOST_AUTO_TEST_CASE( SpinlockBenchmark )
{
	Duration durationMutex;
	{
		int v = 0;
		boost::mutex mutex;
		Time start = Time::now();
		for(int i=0; i<10000000; ++i)
		{
			mutex.lock();
			v+=i;
			mutex.unlock();
		}
		durationMutex = Time::now() - start;
		std::cout << "Mutex took:    " << durationMutex << std::endl;
	}

	Duration durationSpinlock;
	{
		int v = 0;
		Spinlock mutex;
		Time start = Time::now();
		for(int i=0; i<10000000; ++i)
		{
			mutex.lock();
			v+=i;
			mutex.unlock();
		}
		durationSpinlock = Time::now() - start;
		std::cout << "Spinlock took: " << durationSpinlock << std::endl;
	}

	BOOST_CHECK(durationSpinlock < durationMutex);
}
