/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file MetaSerializerTest.C
 *
 * @author Erik Einhorn
 */

#include <iostream>
#include <boost/test/unit_test.hpp>
#include <serialization/MetaSerializer.h>
#include <serialization/IsNotMetaSerializable.h>
#include <serialization/adapters/std/vector>
#include <serialization/adapters/std/list>
#include <serialization/adapters/std/map>
#include <serialization/adapters/std/set>
#include <serialization/adapters/std/pair>
#include <serialization/adapters/boost/tuple.hpp>

#include <utils/UUID.h>
#include <transform/Pose.h>
#include <geometry/Rect.h>

using namespace mira;

class Dimmy
{
public:
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("f", f, "");
		r.member("s", s, "");
	}
	float f;
	std::string s;
};

class Dummy
{
public:

	Dummy() {
		e = NULL;
		f = NULL;
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("a", a, "");
		r.member("b", b, "");
		r.member("c", c, "");
		r.member("d", d, "");
		r.member("e", e, "");
		r.member("f", f, "");
		r.member("g", g, "");
		r.member("h", h, "");
		r.member("i", i, "");
		r.member("j", j, "");
		r.member("k", k, "");
		r.member("l", l, "");
		r.interface("IDummy");
		r.method("setDummy", &Dummy::setDummy, this, "Set a dummy");
	}

	float setDummy(std::vector<Rect2f>* dimmy)
	{
		return 0.0f;
	}

	std::list<int> a;
	std::vector<std::list<float>> b;
	std::map<int,Dimmy> c;
	std::list<Dimmy*> d;
	std::list<Dimmy>* e;
	std::list<Dimmy*>* f;
	std::set<std::string> g;
	float h[5];
	UUID i;
	Pose2 j;
	Pose3 k;
	boost::tuple<int, float, std::string, Dimmy> l;
};

class NotSupported
{
public:
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("a", a, "");
	}

	int a;
};

class NotSupportedRecursive
{
public:
	NotSupportedRecursive()
	{
		a.push_back(NotSupported());
	}
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("a", a, "");
	}

	std::vector<NotSupported> a;
};

namespace mira {
template<>
class IsNotMetaSerializable<NotSupported> : public std::true_type {};
}


BOOST_AUTO_TEST_CASE( MetaSerializerTest )
{
	MetaTypeDatabase db;
	MetaSerializer ms(db);

	std::vector<Dummy> dummy;
	ms.addMeta(dummy);

	BOOST_CHECK_EQUAL(db.size(), 9);
	BOOST_CHECK(db.count("Dimmy") > 0);
	BOOST_CHECK(db.count("Dummy") > 0);

	CompoundMetaPtr dummyMeta = db.find("Dummy")->second;
	BOOST_CHECK_EQUAL(dummyMeta->interfaces.size(), 1);
	BOOST_CHECK(dummyMeta->interfaces.front() == "IDummy");
	BOOST_CHECK_EQUAL(dummyMeta->methods.size(), 1);
	BOOST_CHECK(dummyMeta->methods.front()->name == "setDummy");
	BOOST_CHECK_EQUAL(dummyMeta->methods.front()->returnType->getTypename(), std::string("float"));
	BOOST_CHECK_EQUAL(dummyMeta->methods.front()->parameters.size(), 1);
	BOOST_CHECK_EQUAL(dummyMeta->members.size(), 12);
	// Add more test cases on types and qualifiers if needed

	// print out database for manual debugging purposes
	foreach(auto p, db)
	{
		std::cout << p.first <<  p.second->toString() << std::endl;
	}
}

BOOST_AUTO_TEST_CASE( MetaSerializerNotSupportedTest )
{
	MetaTypeDatabase db;
	MetaSerializer ms(db);

	NotSupported ns;
	BOOST_CHECK_EQUAL(ms.addMeta(ns), TypeMetaPtr());

	NotSupportedRecursive nsr;
	BOOST_CHECK_EQUAL(ms.addMeta(nsr), TypeMetaPtr());
}
