/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file GeometryTest.C
 *    Some geometry test cases.
 *
 * @author Tim Langner
 * @date   2010/10/24
 */

#include <boost/test/unit_test.hpp>
#include <serialization/XMLSerializer.h>
#include <geometry/Geometry.h>

using namespace mira;

///////////////////////////////////////////////////////////////////////////////

BOOST_AUTO_TEST_CASE( PointTest )
{
	Point2f point2f(1.0f, 2.0f);
	Point3f point3f(1.0f, 2.0f, 3.0f);
	Point<float,4> point4f;
	point4f << 1,2,3,4;

	BOOST_CHECK(point2f == Point2f(cv::Point_<float>(1.0f, 2.0f)));

	XMLDom xml;
	XMLSerializer xmls(xml);
	xmls.serialize("P2", point2f);
	xmls.serialize("P3", point3f);
	xmls.serialize("P4", point4f);

	xml.saveToFile("out.xml");

	Point2f point2fd;
	Point3f point3fd;
	Point<float,4> point4fd;
	XMLDeserializer xmlds(xml);
	xmlds.deserialize("P2", point2fd);
	xmlds.deserialize("P3", point3fd);
	xmlds.deserialize("P4", point4fd);

	BOOST_CHECK(point2f == point2fd);
	BOOST_CHECK(point3f == point3fd);
	BOOST_CHECK(point4f == point4fd);
}
///////////////////////////////////////////////////////////////////////////////

BOOST_AUTO_TEST_CASE( SizeTest )
{
	Size2f size2f(1.0f, 2.0f);
	Size3f size3f(1.0f, 2.0f, 3.0f);

	BOOST_CHECK_CLOSE(size2f.width(), 1.0f, 0.00001f);
	BOOST_CHECK_CLOSE(size2f.height(), 2.0f, 0.00001f);

	BOOST_CHECK_CLOSE(size3f.width(), 1.0f, 0.00001f);
	BOOST_CHECK_CLOSE(size3f.height(), 2.0f, 0.00001f);
	BOOST_CHECK_CLOSE(size3f.depth(), 3.0f, 0.00001f);

	BOOST_CHECK(size2f == Size2f(Point2f(1.0f, 2.0f)));
	BOOST_CHECK(size2f == Size2f(cv::Size_<float>(1.0f, 2.0f)));
	BOOST_CHECK(size3f == Size3f(Point3f(1.0f, 2.0f, 3.0f)));

	XMLDom xml;
	XMLSerializer xmls(xml);
	xmls.serialize("S2", size2f);
	xmls.serialize("S3", size3f);

	Size2f size2fd;
	Size3f size3fd;
	XMLDeserializer xmlds(xml);
	xmlds.deserialize("S2", size2fd);
	xmlds.deserialize("S3", size3fd);

	BOOST_CHECK(size2f == size2fd);
	BOOST_CHECK(size3f == size3fd);
}
///////////////////////////////////////////////////////////////////////////////

BOOST_AUTO_TEST_CASE( RectTest )
{
	Rect2f rect2f(Point2f(1.0f, 2.0f), 1.0f, 2.0f);
	Box3f rect3f(Point3f(1.0f, 2.0f, 3.0f), 1.0f, 2.0f, 3.0f);

	BOOST_CHECK(rect2f == Rect2f(Point2f(1.0f, 2.0f), Point2f(2.0f, 4.0f)));
	BOOST_CHECK(rect2f == Rect2f(Point2f(2.0f, 4.0f), Point2f(1.0f, 2.0f)));
	BOOST_CHECK(rect2f == Rect2f(Point2f(1.0f, 2.0f), Size2f(1.0f, 2.0f)));
	BOOST_CHECK(rect2f == Rect2f(boost::geometry::model::box<Point2f>(Point2f(1.0f, 2.0f),
	                                                                  Point2f(2.0f, 4.0f))));
	BOOST_CHECK(rect2f == Rect2f(cv::Rect_<float>(1.0f, 2.0f,1.0f, 2.0f)));

	XMLDom xml;
	XMLSerializer xmls(xml);
	xmls.serialize("R2", rect2f);
	xmls.serialize("R3", rect3f);

	xml.saveToFile("Test.xml");

	Rect2f rect2fd;
	Box3f rect3fd;
	XMLDeserializer xmlds(xml);
	xmlds.deserialize("R2", rect2fd);
	xmlds.deserialize("R3", rect3fd);

	BOOST_CHECK(rect2f == rect2fd);
	BOOST_CHECK(rect3f == rect3fd);
}

BOOST_AUTO_TEST_CASE( RectTestInvalid )
{
	Rect2f rect2f(Point2f(1.0f, 2.0f), 1.0f, 2.0f);
	Box3f rect3f(Point3f(1.0f, 2.0f, 3.0f), 1.0f, 2.0f, 3.0f);

	// recognize valid/invalid

	// points are reordered internally, so it is impossible to create an invalid rect from 2 points
	Rect2f rect2fInvalidPoints(Point2f(1.0f, 2.0f), Point2f(0.0f, 2.0f));
	BOOST_CHECK(rect2fInvalidPoints.isValid());

	Rect2f rect2fInvalidSize(Point2f(1.0f, 2.0f), -1.0f, 2.0f);
	BOOST_CHECK(!rect2fInvalidSize.isValid());

	Box3f rect3fInvalidPoints(Point3f(1.0f, 2.0f, 3.0f), Point3f(1.0f, 2.0f, 0.0f));
	BOOST_CHECK(rect3fInvalidPoints.isValid());

	Box3f rect3fInvalidSize(Point3f(1.0f, 2.0f, 3.0f), 1.0f, 2.0f, -1.0f);
	BOOST_CHECK(!rect3fInvalidSize.isValid());

	// union / intersection invalid with invalid -> invalid

	BOOST_CHECK((Rect2f::invalid() & Rect2f::invalid()) == Rect2f::invalid());
	BOOST_CHECK((Rect2f::invalid() | Rect2f::invalid()) == Rect2f::invalid());

	{
		Rect2f r(Rect2f::invalid());
		r &= Rect2f::invalid();
		BOOST_CHECK(r == Rect2f::invalid());
		r |= Rect2f::invalid();
		BOOST_CHECK(r == Rect2f::invalid());
	}

	BOOST_CHECK((Box3f::invalid() & Box3f::invalid()) == Box3f::invalid());
	BOOST_CHECK((Box3f::invalid() | Box3f::invalid()) == Box3f::invalid());

	{
		Box3f r(Box3f::invalid());
		r &= Box3f::invalid();
		BOOST_CHECK(r == Box3f::invalid());
		r |= Box3f::invalid();
		BOOST_CHECK(r == Box3f::invalid());
	}

	// intersection valid with invalid -> invalid

	BOOST_CHECK((rect2f & Rect2f::invalid()) == Rect2f::invalid());
	BOOST_CHECK((Rect2f::invalid() & rect2f) == Rect2f::invalid());

	{
		Rect2f r(rect2f);
		r &= Rect2f::invalid();
		BOOST_CHECK(r == Rect2f::invalid());
		r &= rect2f;
		BOOST_CHECK(r == Rect2f::invalid());
	}

	BOOST_CHECK((rect3f & Box3f::invalid()) == Box3f::invalid());
	BOOST_CHECK((Box3f::invalid() & rect3f) == Box3f::invalid());

	{
		Box3f r(rect3f);
		r &= Box3f::invalid();
		BOOST_CHECK(r == Box3f::invalid());
		r &= rect3f;
		BOOST_CHECK(r == Box3f::invalid());
	}

	// union valid with invalid -> valid

	BOOST_CHECK((rect2f | Rect2f::invalid()) == rect2f);
	BOOST_CHECK((Rect2f::invalid() | rect2f) == rect2f);

	{
		Rect2f r(Rect2f::invalid());
		r |= rect2f;
		BOOST_CHECK(r == rect2f);
		r |= Rect2f::invalid();
		BOOST_CHECK(r == rect2f);
	}

	BOOST_CHECK((rect3f | Box3f::invalid()) == rect3f);
	BOOST_CHECK((Box3f::invalid() | rect3f) == rect3f);

	{
		Box3f r(Box3f::invalid());
		r |= rect3f;
		BOOST_CHECK(r == rect3f);
		r |= Box3f::invalid();
		BOOST_CHECK(r == rect3f);
	}
}


