/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file BresenhamTest.C
 *    A test case for the Bresenham Algorithm.
 *
 * @author Erik Einhorn
 * @date   2010/12/06
 */

#include <iostream>
#include <boost/test/unit_test.hpp>
#include <geometry/Bresenham.h>

using namespace std;
using namespace mira;

///////////////////////////////////////////////////////////////////////////////

struct BresenhamLineElement
{
	int x, y, length;
};

///////////////////////////////////////////////////////////////////////////////

enum {
	TEST_NORMAL=0x01, // the standard bresenham algorithm
	TEST_RUNSLICE=0x02 // the faster run-slice algorithm
};

///////////////////////////////////////////////////////////////////////////////

// here the ground truths for some lines are defined
struct BresenhamLineGroundTruth
{
	int x0, y0, x1, y1;
	int testFlags;
	int groundTruthSize;
	BresenhamLineElement groundTruth[32];
};

///////////////////////////////////////////////////////////////////////////////

BresenhamLineGroundTruth lineTests[] = {
	////////////////////
	{1,1, 8,3, TEST_NORMAL | TEST_RUNSLICE,
		3,
		{
			{1,1, 2},
			{3,2, 4},
			{7,3, 2}
		}
	},
	////////////////////
	{3,14, 17,11, TEST_RUNSLICE,
		4,
		{
			{15,11, 3},
			{11,12, 4},
			{6,13,  5},
			{3,14,  3}
		}
	},
	////////////////////
	{25,7, 33,9, TEST_NORMAL | TEST_RUNSLICE,
		3,
		{
			{25,7, 2},
			{27,8, 4},
			{31,9, 3}
		}
	},
};

///////////////////////////////////////////////////////////////////////////////

// this class mainly provides the implementations for the visitors
// the visitors checks if the current scanline is equal to the ground truth
class BresenhamLineTester
{
public:

	// visitor for run slice version
	void operator()(int x, int y, int length)
	{
		if(i>=l.groundTruthSize) {
			passed = false;
			cout << "too many segments! current segment: ("
			     << x << ", " << y << ", " << length << ")" << endl;
		}

		const BresenhamLineElement& e = l.groundTruth[i];
		if(e.x!=x || e.y!=y || e.length!=length) {
			passed = false;
			cout << "wrong segment! is: ("
			     << x << ", " << y << ", " << length
			     << "), should: ("
			     << e.x << ", " << e.y << ", " << e.length << ")" << endl;
		}
		++i;
	}

	// visitor for standard bresenham algorithm
	void operator()(int x, int y)
	{
		if(i>=l.groundTruthSize) {
			passed = false;
			cout << "too many segments! " << endl;
		}

		const BresenhamLineElement& e = l.groundTruth[i];

		if(e.x+j != x || e.y!=y) {
			passed = false;
			cout << "wrong pixel! is: ("
			     << x << ", " << y
			     << "), should: ("
			     << e.x+j << ", " << e.y <<  endl;
		}

		// advance our counters
		++j;
		if(j>=l.groundTruth[i].length) {
			++i;
			j=0;
		}
	}

	// constructor of visitors: initialize ground truth and line and row counters
	BresenhamLineTester(const BresenhamLineGroundTruth& il)
	{
		l = il;
		i = 0;
		j = 0;
		passed = true;
	}

public:

	bool passed;
	BresenhamLineGroundTruth l;
	int i, j;
};

///////////////////////////////////////////////////////////////////////////////

// the main testing function
void testVisitors(const BresenhamLineGroundTruth& l)
{
	if(l.testFlags & TEST_NORMAL)
	{
		cout << "Testing Line: " << l.x0 << ", "<< l.y0 << " - "<< l.x1 << ", ";
		cout << l.y1 << endl;
		BresenhamLineTester t(l); // class with correct visitor
		// here the operator (x,y) from object t is used
		bresenham(l.x0,l.y0, l.x1,l.y1, t);
		BOOST_CHECK(t.passed);
	}
	if(l.testFlags & TEST_RUNSLICE)
	{
		cout << "Testing Line with RunSlice: " << l.x0 << ", " << l.y0 << " - ";
		cout << l.x1 << ", " << l.y1 << endl;
		BresenhamLineTester t(l); // class with correct visitor.
		// NOTE: a different operator (x,y,length) is used
		bresenhamRunSlice(l.x0,l.y0, l.x1,l.y1, t);
		BOOST_CHECK(t.passed);
	}
}

///////////////////////////////////////////////////////////////////////////////

BOOST_AUTO_TEST_CASE( VisitorTest )
{
	for(size_t i = 0; i<sizeof(lineTests)/sizeof(BresenhamLineGroundTruth); ++i)
		testVisitors(lineTests[i]);
}
