/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PackageFinder.C
 *    Implementation of PackageFinder.h.
 *
 * @author Tim Langner
 * @date   2013/04/29
 */

#include <utils/PackageFinder.h>
#include <serialization/BinarySerializer.h>
#include <serialization/adapters/std/map>
#include <serialization/adapters/std/vector>
#include <utils/PathFinder.h>
#include <utils/Foreach.h>
#include <utils/Singleton.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

class PackageCache : public LazySingleton<PackageCache>
{
public:

	struct PackageInfo
	{
		template<typename Reflector>
		void reflect(Reflector& r)
		{
			r.member("PackageFile", pkgFile, "Name of package file");
		}

		Path pkgFile;
	};

	struct Cache
	{
		template<typename Reflector>
		void reflect(Reflector& r)
		{
			int v = r.version(1);
			if (v >= 1)
				r.member("MIRAPaths", miraPaths, "List of MIRA project dirs");
			r.member("Time", reindexTime, "Time of reindexing");
			r.member("Packages", packages, "Package list");
		}

		Time reindexTime;
		PathVector miraPaths;
		std::map<std::string, PackageInfo> packages;
	};

	PackageCache();

	bool existsPackage(const std::string& packageName, Duration maxAge = Duration::infinity());
	PackageInfo getPackageInfo(const std::string& packageName, Duration maxAge = Duration::infinity());
	std::map<std::string, Path> getPackagePaths(Duration maxAge = Duration::infinity());

private:

	void readCache(Duration maxAge = Duration::infinity());

	bool checkCache(Duration maxAge = Duration::infinity())
	{
		if ((Time::now() - mCache.reindexTime) > maxAge)
			return false;
		if (mMiraPaths.size() != mCache.miraPaths.size())
			return false;
		for(std::size_t i=0; i<mMiraPaths.size(); ++i)
			if (mMiraPaths[i] != mCache.miraPaths[i])
				return false;
		return true;
	}

	bool checkForPackage(const std::string& packageName, Duration maxAge = Duration::infinity());

	void reindex();

	Cache mCache;
	boost::mutex mCacheMutex;
	PathVector mMiraPaths;
};

///////////////////////////////////////////////////////////////////////////////

PackageCache::PackageCache()
{
	mMiraPaths = getMIRAPaths();
	std::sort(mMiraPaths.begin(), mMiraPaths.end());
	readCache();
}

bool PackageCache::existsPackage(const std::string& packageName, Duration maxAge)
{
	boost::mutex::scoped_lock lock(mCacheMutex);
	return checkForPackage(packageName, maxAge);
}

PackageCache::PackageInfo PackageCache::getPackageInfo(const std::string& packageName, Duration maxAge)
{
	boost::mutex::scoped_lock lock(mCacheMutex);
	if (!checkForPackage(packageName, maxAge))
		MIRA_THROW(XInvalidConfig, "Package '" << packageName << "' does not exist.");
	return mCache.packages[packageName];
}

std::map<std::string, Path> PackageCache::getPackagePaths(Duration maxAge)
{
	std::map<std::string, Path> ret;
	boost::mutex::scoped_lock lock(mCacheMutex);

	// check if cache is still valid
	Time expiredTime = mCache.reindexTime + maxAge;
	if(Time::now()>expiredTime)
		reindex();

	auto it = mCache.packages.begin();
	for(; it != mCache.packages.end(); ++it)
		ret[it->first] = it->second.pkgFile.parent_path();
	return ret;
}

void PackageCache::readCache(Duration maxAge)
{
	boost::mutex::scoped_lock lock(mCacheMutex);
	try
	{
		Path cacheFile = getAppDataDirectory() / "package.cache";
		std::ifstream ifs(cacheFile.string().c_str());
		BinaryStlIstream bis(ifs);
		BinaryStreamDeserializer deserializer(bis);
		deserializer.deserialize(mCache);
		if (!checkCache(maxAge))
			reindex();
	}
	catch(...)
	{
		reindex();
	}
}

bool PackageCache::checkForPackage(const std::string& packageName, Duration maxAge)
{
	auto it = mCache.packages.find(packageName);
	if (it == mCache.packages.end() ||
		!boost::filesystem::exists(it->second.pkgFile))
	{
		reindex();
		it = mCache.packages.find(packageName);
	}
	return it != mCache.packages.end();
}

void PackageCache::reindex()
{
	PathVector packgeFiles = findProjectFiles("*.package", true);
	mCache.packages.clear();
	mCache.reindexTime = Time::now();
	mCache.miraPaths = getMIRAPaths();
	std::sort(mCache.miraPaths.begin(), mCache.miraPaths.end());
	foreach(const Path& p, packgeFiles)
	{
		std::string pkgName = p.stem().string();
		mCache.packages[pkgName].pkgFile = p;
	}
	Path cacheFile = getAppDataDirectory() / "package.cache";
	std::ofstream ofs(cacheFile.string().c_str());
	BinaryStlOstream bos(ofs);
	BinaryStreamSerializer serializer(bos);
	serializer.serialize(mCache);
}

///////////////////////////////////////////////////////////////////////////////

Path findPackage(const std::string& package, Duration maxCacheAge)
{
	PackageCache::PackageInfo pkg = PackageCache::instance().getPackageInfo(package, maxCacheAge);
	return pkg.pkgFile.parent_path();
}

std::map<std::string, Path> findPackages(Duration maxCacheAge)
{
	return PackageCache::instance().getPackagePaths(maxCacheAge);
}

///////////////////////////////////////////////////////////////////////////////

} // namespace

