/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Factory.C
 *    $Implementation of Factory.h$.
 *
 * @author Ronny Stricker
 * @date   2010/11/03
 */

#include <factory/Factory.h>
#include <factory/VacantClass.h>

namespace mira {

//////////////////////////////////////////////////////////////////////////////

mira::ClassFactory::~ClassFactory()
{
	while( mClasses.size() > 0 ) {
		unregisterClass( mClasses.begin()->second.get() );
	}
}

void mira::ClassFactory::registerClass( boost::shared_ptr<Class> iClass )
{
	assert( iClass.get() );

	boost::recursive_mutex::scoped_lock lock( instance().mThreadMutex );
	instance().internalClassRegister( iClass );
}

void mira::ClassFactory::registerClass(
		boost::shared_ptr<Class> iClass, boost::shared_ptr<Class> baseClass )
{
	assert( iClass.get() );
	// identifiers of base and parent have to be different.
	// otherwise this function will end up in an infinite loop
	// if both identifiers are identical, the object macro seems to be missing
	// for the child class
	if ( iClass->getIdentifier() == baseClass->getIdentifier() ) {
		MIRA_THROW( XLogical, "BaseClass and Class do have the same identifiers ("
		            + iClass->getIdentifier() + ")! Maybe you forgot the factory"
		            "  macro in a derived class of " + iClass->getIdentifier()
		            + "?" );
	}

	boost::recursive_mutex::scoped_lock lock( instance().mThreadMutex );
	if ( instance().internalClassRegister( iClass ) ) {
		// we have to assure that the base class is registered at the factory
		// in order to add the class proxy of the parents safely
		if ( !instance().isClassRegistered( baseClass->getIdentifier() ) ) {
			// there is no need to call register class with the real parents at
			// this point since it will be called automatically later one (by
			// the register class macro)
			registerClass( baseClass );
		}
		iClass->mDirectParents[ baseClass->getIdentifier() ] =
				ClassProxy( &(instance().mClasses[baseClass->getIdentifier()]) );

		ClassProxy tChildProxy( &(instance().mClasses[iClass->getIdentifier()]) );
		instance().propagateChild(tChildProxy, *baseClass.get());
	}
}

void mira::ClassFactory::unregisterClass( Class* iClass )
{
	if ( iClass && iClass->getIdentifier() != "mira::Object" ) {
		iClass->eraseChild( iClass );
		iClass->eraseParent( iClass );

		// it is extremely unsave to have access to a class proxy if we
		// don't even know if the class factory exists -> remove all
		// class proxy objects from the class
		iClass->mDerivedChildren.clear();
		iClass->mDirectParents.clear();

		if ( !isDestroyed() && instance().isClassRegistered( iClass->getIdentifier() ) ) {
			boost::recursive_mutex::scoped_lock lock( instance().mThreadMutex );
			instance().mRoot.eraseChild( iClass );

			std::map<std::string, boost::shared_ptr<Class> >::const_iterator it = instance().mClasses.find( iClass->getIdentifier() );
			if ( it != instance().mClasses.end() && it->second.get() == iClass )
				instance().mClasses.erase( iClass->getIdentifier() );
		}
	}
}

void ClassFactory::postRegisterBaseClasses(std::string const& iClass,
                                           std::vector< std::string> const& parents )
{
	boost::recursive_mutex::scoped_lock lock( instance().mThreadMutex );
	try {
		// get class object of child class
		ClassProxy tChildClass = instance().mRoot.getClassByIdentifier( iClass );
		Class* tChildPtr = tChildClass.mClass->get();

		// cycle through parents of child
		foreach ( std::string tParent, parents ) {
			// get class object of parent
			ClassProxy tParentClass = instance().mRoot.getClassByIdentifier( tParent);
			tChildPtr->mDirectParents[tParent] = tParentClass;

			Class* tParentPtr = tParentClass.mClass->get();

			// add child to list of parents if necessary
			if ( tParentPtr->mDerivedChildren.find( iClass ) ==
					tParentPtr->mDerivedChildren.end() ) {
				tParentPtr->mDerivedChildren[ iClass ] = tChildClass;
			}
		}
	}
	catch (XInvalidParameter& e ) {
		MIRA_RETHROW(e, "in postRegisterBaseClasses");
	}
}

void ClassFactory::finalizePostRegister()
{
	boost::recursive_mutex::scoped_lock lock( instance().mThreadMutex );
	// cycle through all classes and propagate all child classes to the parents
	foreach( auto tClass, instance().mClasses ) {
		foreach( auto tParent, tClass.second->mDirectParents ) {
			// if the following assertion is not met, you forgot to register
			// one or more classes. This may be the case if not all necessary
			// manifest files are loaded.
			assert( instance().isClassRegistered( tClass.first ) );
			ClassProxy tChildProxy(&instance().mClasses[tClass.first]);
			Class* tParentPtr = tParent.second.mClass->get();
			instance().propagateChild( tChildProxy, *(tParentPtr) );
		}
	}
}

void ClassFactory::propagateChild( ClassProxy& child, Class& parent )
{
	Class* tChildPtr = child.mClass->get();

	if ( parent.mDerivedChildren.find( tChildPtr->getIdentifier() ) ==
			parent.mDerivedChildren.end() ) {
		parent.mDerivedChildren[ tChildPtr->getIdentifier() ] =
				child;
	}
	parent.mDerivedChildren.insert( tChildPtr->mDerivedChildren.begin(),
			tChildPtr->mDerivedChildren.end() );

	// add meta information of parent
	tChildPtr->mMetaInfo.insert(parent.getMetaInfo().begin(),
	                            parent.getMetaInfo().end());
	// recursively cycle trough next parents
	typedef std::map<std::string, ClassProxy > mapType;
	foreach( mapType::value_type tParent, parent.mDirectParents ) {
		Class* tParentPtr = tParent.second.mClass->get();
		propagateChild( child, *tParentPtr );
	}

}

bool ClassFactory::internalClassRegister( boost::shared_ptr<Class> iClass )
{
	assert( iClass.get() );
	// Try to register the given class. It is not allowed to register a
	// identifier twice. However, there are two exceptions. It is allowed to
	// overwrite a Vacant Class with a TClass Object. Furthermore, the
	// attempt to overwrite a TClass with a Vacant class will be silently
	// ignored (this is the case if a program is linked against a certain
	// library whose manifest file is loaded later on).
	if ( mClasses.find( iClass->getIdentifier() ) != mClasses.end() ) {

		Class* tRegClass = mClasses[iClass->getIdentifier()].get();
		if ( ( tRegClass->isLibraryLoaded() && tRegClass != iClass.get() ) ||
				( !tRegClass->isLibraryLoaded() && !iClass->isLibraryLoaded() ) )
		{
			// ignore the attempt to overwrite TClass with vacant class
			if ( tRegClass->isLibraryLoaded() && !iClass->isLibraryLoaded() ) {
				// however, we should save the lib name
				if ( !tRegClass->mLib.empty() && tRegClass->mLib != iClass->mLib ) {
					// that is strange! Registered class does already know
					// its library name but is is different!?

					MIRA_THROW( XLogical, "Class identifier already in use ("
								+ iClass->getIdentifier()
								+ ")! First occurrence "
								+ resolveLibraryName( tRegClass->mLib ).string()
								+ " Second occurrence "
								+ resolveLibraryName( iClass->mLib ).string() );
				}
				tRegClass->mLib = iClass->mLib;
				return false;
			}
			MIRA_THROW( XLogical, "Class identifier already in use ("
						+ iClass->getIdentifier()
						+ ")! First occurrence "
						+ resolveLibraryName( tRegClass->mLib ).string()
						+ " Second occurrence "
						+ resolveLibraryName( iClass->mLib ).string() );
		}

		// we have to take care of the registered child classes.
		// The new TClass Object perhaps don't know all of its child
		// -> copy all unknown child to the new class object
		foreach(auto ChildIt, tRegClass->mDerivedChildren ) {
			if ( iClass->mDerivedChildren.find( ChildIt.first ) ==
					iClass->mDerivedChildren.end() ) {
				iClass->mDerivedChildren[ ChildIt.first ] = ChildIt.second;
			}
		}
	}

	mClasses[iClass->getIdentifier()] = iClass;

	mRoot.mDerivedChildren[iClass->getIdentifier()] =
			ClassProxy( &(mClasses[iClass->getIdentifier()]) );
	return true;
}

//////////////////////////////////////////////////////////////////////////////

} // namespace
