/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file UUID.h
 *    Typedefs and serialization support for uuids.
 *
 * @author Tim Langner
 * @date   2010/10/04
 */

#ifndef _MIRA_UUID_H_
#define _MIRA_UUID_H_

#ifndef Q_MOC_RUN
#include <boost/uuid/uuid.hpp>
#include <boost/uuid/uuid_generators.hpp>
#include <boost/uuid/uuid_io.hpp>
#endif

#include <serialization/IsAtomicSerializable.h>
#include <stream/BinaryStream.h>
#include <utils/ToString.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/// Shorter name for boost uuid
typedef boost::uuids::uuid UUID;

/// binary stream operators
template <typename StreamUnderlay>
BinaryOstream<StreamUnderlay>& operator<<(BinaryOstream<StreamUnderlay>& s,
                                          const UUID& uuid)
{
	const char* buffer = reinterpret_cast<const char*>(uuid.data);
	s.write(buffer, UUID::static_size());
	return s;
}

/// binary stream operators
template <typename StreamUnderlay>
BinaryIstream<StreamUnderlay>& operator>>(BinaryIstream<StreamUnderlay>& s,
                                          UUID& uuid)
{
	char* buffer = reinterpret_cast<char*>(uuid.data);
	s.read(buffer, UUID::static_size());
	return s;
}

template<>
class IsAtomicSerializable<UUID> : public std::true_type {};

/// specialization for toString
template<>
inline std::string toString<UUID>(const UUID& value, int precision)
{
	std::stringstream ss;
	ss << value;
	return ss.str();
}

/// specialization for fromString
template <>
inline UUID fromString<UUID>(const std::string& str)
{
	std::stringstream ss(str);
	UUID uuid;
	ss >> uuid;
	return uuid;
}

///////////////////////////////////////////////////////////////////////////////

}

#endif
