/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file XMLSerializer.h
 *	XMLSerializer and XMLDeserializer.
 *
 * @author Erik Einhorn
 * @date   2010/07/01
 */

#ifndef _MIRA_XMLSERIALIZER_H_
#define _MIRA_XMLSERIALIZER_H_

#include <iostream>
#include <list>
#include <algorithm>

#include <serialization/Serializer.h>
#include <serialization/Deserializer.h>

#include <xml/XMLDom.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

///@cond INTERNAL
namespace Private {

inline std::string replaceSpaces(const std::string& s)
{
	std::string tag = s;
	for(std::size_t i=0; i<tag.size(); ++i)
	{
		if(tag[i]==' ')
			tag[i]='_';
	}
	return tag;
}

} // namespace

///@endcond

/**
 * @ingroup SerializationModule
 *
 * Serializer for serializing objects in XML format.
 * It uses the XMLDom class for storing the XML content.
 *
 * Example usage:
 * \code
 * #include <serialization/adapters/std/vector>
 * // the data to be serialized
 * std::vector<int> myValue;
 * ...
 *
 * // create a XMLSerializer that writes into an XML document
 * XMLDom myXmlDocument;
 * XMLSerializer serializer(myXmlDocument);
 *
 * // serialize the data
 * serializer.serialize("myValue", myValue, "A comment describing the value");
 *
 * // write the XML document to a file
 * myXmlDocument.saveToFile("myfile.xml");
 * \endcode
 *
 * For the deserialization of XML content see
 * @ref mira::XMLDeserializer "XMLDeserializer".
 *
 * @see @ref SerializationPage
 */

struct XMLSerializerTag {};

class XMLSerializer : public Serializer<XMLSerializer>
{
	typedef Serializer<XMLSerializer> Base;

public:
	typedef XMLSerializerTag Tag;

	/// Flags for XML output format
	enum OutputFormat
	{
		STANDARD      = 0x00, ///< Default format
		NO_COMMENTS   = 0x01, ///< No comments are written to output document
		COMPRESSED    = 0x02, ///< atomic members are written as attributes instead of childs
	};

	MIRA_ENUM_TO_FLAGS_INCLASS(OutputFormat) // generate logical operators for enum values

	/**
	 * Create a new XML serialization on top of the passed XMLDom document.
	 * @note: The XMLDom object must exist as long as the serializer is used.
	 * @param[in] iXmlDom The document used for output
	 * @param[in] format Combination of output format flags
	 */
	XMLSerializer(XMLDom& iXmlDom, OutputFormat format = STANDARD) :
		mRoot(iXmlDom.root()), mNode(iXmlDom.root()), mOutputFormat(format)
	{
	}

	/**
	 * Create a new XML serialization on top of the passed XMLDom iterator node.
	 * @note: The XMLDom document must exist as long as the serializer is used.
	 * @param[in] iRoot The root node of the document used for output
	 * @param[in] format Combination of output format flags
	 */
	XMLSerializer(XMLDom::iterator& iRoot, OutputFormat format = STANDARD) :
		mRoot(iRoot), mNode(iRoot), mOutputFormat(format)
	{
	}

	int version(int version)
	{
		// add version
		mNode.add_attribute("version", toString(version) );
		return version;
	}

	template<typename T>
	void atomic(T& member)
	{
		const ReflectMemberMeta& meta = getCurrentMemberMeta();
		if (mOutputFormat & COMPRESSED)
			mNode.add_attribute(Private::replaceSpaces(meta.name), toString(member,12));
		else
		{
			if(!(mOutputFormat & NO_COMMENTS) && meta.comment!=NULL && strlen(meta.comment)>0)
				mNode.add_comment(meta.comment);
			mNode.add_child(Private::replaceSpaces(meta.name)).add_content(toString(member,12));
		}
	}

	template<typename T>
	void object(T& member)
	{
		const ReflectMemberMeta& meta = getCurrentMemberMeta();

		// create a new node
		if(!(mOutputFormat & NO_COMMENTS) && meta.comment!=NULL && strlen(meta.comment)>0)
			mNode.add_comment(meta.comment);
		XMLDom::iterator i = mNode.add_child(Private::replaceSpaces(meta.name));

		// enter the new node
		mNode = i;

		if(!mPointerClass.empty()) {
			mNode.add_attribute("class", mPointerClass);
			mPointerClass.clear();
		}

		Base::object(member);

		// leave the node
		mNode = mNode.parent();
		assert(mNode!=mRoot.end());

	}

	// Stores a reference to a previously serialized object using the "ref" attribute.
	void pointerReference(int referencedObjectID) {
		const char* name = getCurrentMemberMeta().name;
		mNode.add_child(Private::replaceSpaces(name)).add_attribute("ref", this->getHumanReadableFullID(referencedObjectID));
	}

	// Stores the class type the "class" attribute.
	void pointerWithClassType(const std::string& type) {
		mPointerClass = type;
		//mNode.add_attribute("class", type);
	}

	// Stores a NULL pointer using the "nullptr" attribute.
	void pointerNull() {
		const char* name = getCurrentMemberMeta().name;
		mNode.add_child(Private::replaceSpaces(name)).add_attribute("nullptr", "true");
	}

public:

	/// Accessor for special serialize/deserialize methods below
	XMLDom::iterator getNode() { return mNode; }

private:

	/// iterator to the root node that was passed in the constructor
	XMLDom::iterator mRoot;

	// iterator to the current node
	XMLDom::iterator mNode;

	std::string mPointerClass;

	OutputFormat mOutputFormat;

};

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup SerializationModule
 *
 * Deserializer for serializing objects from XML format.
 * It uses the XMLDom class for storing the XML content.
 *
 * Example usage:
 * \code
 * // load the XML document and create the XMLDeserializer:
 * XMLDom myXmlDocument;
 * myXmlDocument.loadFromFile("myfile.xml");
 * XMLDeserializer deserializer(myXmlDocument);
 *
 * // the object that will be filled with the content
 * std::vector<int> myValue;
 *
 * // deserialize the value
 * deserializer.deserialize("myValue", myValue);
 *
 * // myValue is now filled with the content that was stored in the XML file
 * // within the "myValue" node.
 * \endcode
 *
 * For the serialization of XML content see
 * @ref mira::XMLSerializer "XMLSerializer".
 *
 * @see @ref SerializationPage
 */
class XMLDeserializer : public Deserializer<XMLDeserializer>
{
	typedef Deserializer<XMLDeserializer> Base;
public:

	/**
	 * Create a new XML deserialization on top of the passed XMLDom document.
	 * @note: The XMLDom object must exist as long as the serializer is used.
	 */
	XMLDeserializer(const XMLDom& iXmlDom) :
		mRoot(iXmlDom.root()), mNode(iXmlDom.root())
	{
	}

	/**
	 * Create a new XML deserialization on top of the passed XMLDom iterator node.
	 * @note: The XMLDom document must exist as long as the serializer is used.
	 */
	XMLDeserializer(const XMLDom::const_iterator& iRoot) :
		mRoot(iRoot), mNode(iRoot)
	{
	}


	/// overwritten from Deserializer, to handle exceptions and to add additional
	/// information about file name and line
	template <typename T>
	void deserialize(const std::string& name, T& value)
	{
		mDeserializeFromThisNode=false;
		mNextFindIteratorStack.clear();
		pushFindStack();
		try {
			Base::deserialize(name,value);
		} catch (Exception& ex) {
			MIRA_RETHROW(ex, "in tag starting at: " <<  mNode.uri() << "(" << mNode.line() << ")");
		}
		popFindStack();
	}

	/**
	 * In contrast to the deserialize() method this method will use the node
	 * that was specified in the constructor for deserialzing the given member.
	 */
	template <typename T>
	void deserializeFromNode(const char* name, T& value)
	{
		mDeserializeFromThisNode=true;
		mNextFindIteratorStack.clear();
		pushFindStack();
		try {
			Base::deserialize(name,value);
		} catch (Exception& ex) {
			MIRA_RETHROW(ex, "in tag starting at: " <<  mNode.uri()
			            << "(" << mNode.line() << ")");
		}
		popFindStack();
	}

	int version(int expectedVersion)
	{
		// obtain version
		int version = mNode.get_attribute<int>("version", 0);
		if (version > expectedVersion)
		{
			MIRA_LOG(WARNING) << "Trying to deserialize XML data of a newer version (" << version <<
				") of a type into an older version (" << expectedVersion << ").";
		}
		return version;
	}

	template<typename T>
	void atomic(T& member)
	{
		Base::atomic(member);

		if(mAtomicFromAttribute.empty())
			member = fromString<T>(*mNode.content_begin());
		else // try to get the value from the attribute
			 // (which should exists, since this is checked in invokeMemberOverwrite)
			member = mNode.get_attribute<T>(mAtomicFromAttribute); //uses fromString<T> internally
	}

	void beginTag(const ReflectMemberMeta& meta)
	{
		if(meta.name!=NULL) {
			if(!mAtomicFromAttribute.empty()) {
				MIRA_THROW(XIO, "Cannot deserialize '" << mAtomicFromAttribute <<
				                "' from an attribute as it is a complex type, that "
				                "must be stored as an XML node.");
			}

			XMLDom::const_iterator i = findNext(Private::replaceSpaces(meta.name));
			if(i!=mNode.end()) {
				// found it, so set it as current node
				mNode = i;
			} else {
				// node not found
				MIRA_THROW(XMemberNotFound, "Node '"
				          << Private::replaceSpaces(meta.name)
				          << "' is missing"); // hence we have to abort with an exception
			}
			pushFindStack();
		}
	}

	void endTag(const ReflectMemberMeta& meta)
	{
		if(meta.name!=NULL) {
			assert(mNode!=mRoot); // we will never go above the root node.
			popFindStack();
			mNode = mNode.parent();
			assert(mNode!=mRoot.end());
		}
	}

	/*
	 * Overwrites Deserializer::pointer.
	 * This method checks, if there is a pointer reference specified
	 * (the ref attribute is present). If so, it reads the referenced full id
	 * of the previously stored object and uses resolveReference() to get the
	 * pointer to that previously deserialized object.
	 * Otherwise it calls pointer() of the Deserializer base class to
	 * deserialize the full object.
	 */
	template<typename T>
	void pointer(T* &pointer)
	{
		// do we have a null pointer?
		if(mNode.get_attribute<std::string>("nullptr","false") == "true") {
			pointer = NULL;
			return;
		}

		// do we have a reference?
		std::string ref = mNode.get_attribute<std::string>("ref","");
		if(ref.size()>0) {
			// we have a reference, so resolve it
			pointer = resolveReference<T>(ref);
			return;
		}

		// we have a "normal" pointer, so deserialize it
		Base::pointer(pointer);
	}

	std::string pointerClassType()
	{
		return mNode.get_attribute<std::string>("class","");
	}

	// hijack the invokeOverwrite method for adding a starting and closing tag
	// around the serialized data
	template<typename T>
	void invokeMemberOverwrite(T& member, const ReflectMemberMeta& meta)
	{
		bool fromThisNode = mDeserializeFromThisNode;
		bool fromAttribute = false;

		if(!fromThisNode) {
			try {
				beginTag(meta);
			} catch(XMemberNotFound&) {
				// member of value not found, if it is an atomic type
				// we can still try to get the value from an attribute with
				// the same name
				// (however, the following reserved attributes are
				//  not allowed: version, nullptr, ref, class)
				std::string name = Private::replaceSpaces(meta.name);
				if(name!="version" &&
				   name!="nullptr" &&
				   name!="ref" &&
				   name!="class")
				{
					// check if the attribute exists
					if(!mNode.has_attribute(name)) {
						MIRA_THROW(XMemberNotFound, "Node '" << name <<
							"' is missing and no such attribute was found");
					}
					mAtomicFromAttribute = name;
					fromAttribute = true;
				} else {
					// otherwise rethrow the exception since we cannot do anything
					throw;
				}
			}
		}

		mDeserializeFromThisNode=false;

		try {
			Base::invokeMemberOverwrite(member, meta);
		} catch(...) {
			// make sure our find stack is correctly unwound and the correct
			// node in the XML file is maintained if an exception
			// is thrown somewhere in our child nodes (fixes #536)
			mAtomicFromAttribute.clear();
			if(!fromThisNode && !fromAttribute)
				endTag(meta); // finish this tag and ...
			throw; // ... rethrow
		}

		// hm, this cleanup code is a duplicate of the above cleanup
		// code in the event of an exception
		mAtomicFromAttribute.clear();
		if(!fromThisNode && !fromAttribute)
			endTag(meta);
	}

public: // accessors for special serialize/deserialize methods below

	XMLDom::const_iterator getNode() { return mNode; }

private:

	/// iterator to the root node that was passed in the constructor
	XMLDom::const_iterator mRoot;

	/// iterator to the current node
	XMLDom::const_iterator mNode;

	bool mDeserializeFromThisNode;

	/**
	 * if not empty, then try to get the value of an atomic member from
	 * the attribute of the xml node, since there is no child node with
	 * the name.
	 */
	std::string mAtomicFromAttribute;

private:
	// Necessary to allow multiple nodes with the same name.
	// Here we store the iterators of previous searches for
	// a certain nodes, the next search will then continue
	// after the previous found iterator.
	// Those iterators must be stored in a stack for each
	// level of the XML tree.
	typedef std::map<std::string, XMLDom::const_iterator> NextFindIteratorMap;
	std::list<NextFindIteratorMap> mNextFindIteratorStack;

	XMLDom::const_iterator findNext(const std::string& name)
	{
		assert(mNextFindIteratorStack.size()>=1);
		NextFindIteratorMap& m = mNextFindIteratorStack.back();

		XMLDom::const_iterator i = mNode.end();

		// check if we have searched for the name already
		NextFindIteratorMap::iterator p = m.find(name);
		if(p==m.end()) {
			// no, not yet

			// so search from the beginning
			i = find(mNode.begin(), mNode.end(), name);
		} else {
			// we already searched for "name", so continue search from that point
			i = find(p->second, mNode.end(), name);
		}

		// if we found something, save the successor for next search
		if(i!=mNode.end()) {
			XMLDom::const_iterator j=i;
			++j; // go to successor
			m[name] = j; // and store it for next search
		}

		return i;
	}

	// is called when we enter a new node
	void pushFindStack()
	{
		mNextFindIteratorStack.push_back(NextFindIteratorMap());
	}

	// is called when we leave a node
	void popFindStack()
	{
		mNextFindIteratorStack.pop_back();
	}

};

///@cond INTERNAL

// specializations for STL containers

namespace serialization { // our private namespace

///////////////////////////////////////////////////////////////////////////////

/**
 * Specialization for XML Serializer which does not write the
 * item count explicitly. The deserializer will count the
 * item nodes in the parent node to recover the item count, which
 * is much more user friendly, since the user does not need
 * to provide the count itself.
 */
template<typename Collection>
struct ReflectCollectionCount<XMLSerializer,Collection>
{
	static void reflect(XMLSerializer& r, uint32& ioCount)
	{
		// do nothing
	}
};

/**
 * Specialization for XML Deserializer which counts the
 * item nodes in the parent node to recover the item count, which
 * is much more user friendly, since the user does not need
 * to provide the count itself.
 */
template<typename Collection>
struct ReflectCollectionCount<XMLDeserializer,Collection>
{
	static void reflect(XMLDeserializer& r, uint32& ioCount)
	{
		XMLDom::const_iterator node = r.getNode();

		// count the "item" nodes
		ioCount = (uint32)std::count(node.begin(), node.end(), "item");
	}
};

///////////////////////////////////////////////////////////////////////////////

} // namespace

///@endcond

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
