/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file NoGenericReflect.h
 *    Provides the MIRA_NO_GENERIC_REFLECT macros
 *
 * @author Erik Einhorn
 * @date   2011/08/03
 */

#ifndef _MIRA_NOGENERICREFLECT_H_
#define _MIRA_NOGENERICREFLECT_H_

#include <error/Exceptions.h>
#include <platform/Typename.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup SerializationModule
 * Exception that is thrown, if a certain type does not support the specified
 * reflector/serializer/deserializer since it does not support a generic
 * reflect method that matches with all reflectors.
 */
class XNoGenericReflect : public XNotImplemented
{
public:
	XNoGenericReflect(const std::string& msg,
			          const char* file=NULL, int line=0) throw() :
		XNotImplemented(msg, file, line) {}

};

/**
 * @ingroup SerializationModule
 * Macro that can be used to indicate that a class type does not support
 * all possible Reflectors/Serializers that may visit the reflect method of
 * the type. This is useful for class types that, e.g. support the
 * BinarySerializer only. In this case the reflect method must be specialized
 * for that certain reflector as shown in the following example:
 * \code
 * class MyClass
 * {
 *     // indicate that no generic reflect for all Reflectors is supported
 *     MIRA_NO_GENERIC_REFLECT_MEMBER(MyClass)
 *
 *     template<typename BinaryStream>
 *     void reflect(BinarySerializer<BinaryStream>& r) {
 *         // special code for BinarySerializer
 *     }
 * };
 * \endcode
 */
#define MIRA_NO_GENERIC_REFLECT_MEMBER(Type)                                    \
template<typename Derived>                                                     \
void reflect(mira::ReflectorInterface<Derived>& r)                             \
{                                                                              \
	MIRA_THROW(XNoGenericReflect, "'" << typeName<Type>()                       \
	          << "' does not support the reflector '"                          \
	          << typeName<Derived>() << "'");                                  \
}

/**
 * @ingroup SerializationModule
 * Macro that can be used to indicate that a class type does not support
 * all possible Reflectors/Serializers that may visit the reflect method of
 * the type. This is useful for class types that, e.g. support the
 * BinarySerializer only. In this case the reflect method must be specialized
 * for that certain reflector as shown in the following example:
 * \code
 * // indicate that no generic reflect for all Reflectors is supported
 * MIRA_NO_GENERIC_REFLECT_MEMBER(MyClass)
 *
 * template<typename BinaryStream>
 * void reflect(BinarySerializer<BinaryStream>& r, MyClass& obj) {
 *     // special code for BinarySerializer
 * }
 * \endcode
 *
 * In contrast to MIRA_NO_GENERIC_REFLECT_MEMBER this macro can be used
 * outside of a class.
 */
#define MIRA_NO_GENERIC_REFLECT(Type)                                           \
template<typename Derived>                                                     \
void reflect(mira::ReflectorInterface<Derived>& r, Type& value)                \
{                                                                              \
	MIRA_THROW(XNoGenericReflect, "'" << typeName<Type>()                       \
	          << "' does not support the reflector '"                          \
	          << typeName<Derived>() << "'");                                  \
}

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
