/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file HashStream.h
 *    Some basic classes for different Hash function.
 *
 * @author Christian Martin
 * @date   2010/09/09
 */

#ifndef _MIRA_HASH_STREAM_H_
#define _MIRA_HASH_STREAM_H_

#include <ostream>
#include <string>
#include <vector>

#include <platform/Types.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * A generic hash digest, which consists of an array of bytes.
 *
 * @ingroup SecurityModule
 */
class MIRA_BASE_EXPORT HashDigest :
	public std::vector<uint8>
{
public:
	/**
	 * Constructor
	 * @param[in] iLength The length of the hash digest.
	 */
	HashDigest(size_t iLength) :
		std::vector<uint8>(iLength) {}

	/**
	 * Convert the digest into a string.
	 * @return The hash digest as a string.
	 */
	std::string toString() const;
};

///////////////////////////////////////////////////////////////////////////////

/**
 * @brief A generic hash streambuf class.
 *
 * A specific hash streambuf must inherit from this class and must overload
 * the virtual protected member function xsputn from basic_streambuf.
 *
 * @ingroup SecurityModule
 */
class HashStreamBuf :
	public std::basic_streambuf<char, std::char_traits<char> >
{
public:
	/**
	 * Reset the hash value.
	 */
	virtual void reset() = 0;

	/**
	 * Return the current digest of the hash algorithm.
	 * @return The hash digest.
	 */
	virtual HashDigest getDigest() const = 0;
};

///////////////////////////////////////////////////////////////////////////////

/**
 * A template base class for hash functions based on std::ostream.
 *
 * To use this class, a specialization of this template using a specific
 * implementation of HashStreamBuf has to be used.
 *
 * Usage example:
 * \code
 *
 *     // Implementation of an own HashStreamBuf class
 *     class MyHashStreamBuf :
 *         public HashStreamBuf
 *     {
 *     public:
 *         MyHashStreamBuf() { ... }
 *         ~MyHashStreamBuf() { ... }
 *     public:
 *         // Implementation of HashStreamBuf::reset
 *         virtual void reset() { ... }
 *         // Implementation of HashStreamBuf::getDigest
 *         virtual HashDigest getDigest() const { ... }
 *     protected:
 *         // Implementation of std::basic_streambuf::xsputn
 *         virtual std::streamsize xsputn(const char* s, std::streamsize n) { ... }
 *     };
 *
 *     // Typedef for the hash function
 *     typedef HashStream<MyHashStreamBuf> MyHash;
 *
 *     int main(int pArgc, char* pArgv[])
 *     {
 *         MyHash hash;
 *
 *         hash << "Some data.";
 *         std::cout << "Digest1: " << hash.getDigest().toString() << std::endl;
 *
 *         hash.reset();
 *         hash << "Some other data.";
 *         std::cout << "Digest2: " << hash.getDigest().toString() << std::endl;
 *     }
 *
 * \endcode
 *
 * @ingroup SecurityModule
 */
template<class T>
class HashStream :
	public std::ostream
{
public:
	/// Th default constructor.
	HashStream() : std::ostream(&mBuffer) {}

	/// The destructor.
	virtual ~HashStream() {}

	/**
	 * Reset the hash value.
	 */
	virtual void reset() { mBuffer.reset(); }

	/**
	 * Return the current digest of the hash algorithm.
	 * @return The hash digest.
	 */
	HashDigest getDigest() const { return(mBuffer.getDigest()); }

public:
	/// Put a C-string in the hash stream.
	HashStream& operator<<(const char* value) {
		write(value, strlen(value));
		return *this;
	}

	/// Put a STL string in the hash stream.
	HashStream& operator<<(const std::string& value) {
		write(value.c_str(), value.size());
		return *this;
	}

	/// Put a @c bool in the hash stream.
	HashStream& operator<<(const bool& value) {
		write(&value, sizeof(bool));
		return *this;
	}

	/// Put a @c char in the hash stream.
	HashStream& operator<<(const char& value) {
		write(&value, sizeof(char));
		return *this;
	}

	/// Put a @c uint8 in the hash stream.
	HashStream& operator<<(const uint8& value) {
		write(&value, sizeof(uint8));
		return *this;
	}

	/// Put a @c uint16 in the hash stream.
	HashStream& operator<<(const uint16& value) {
		write(&value, sizeof(uint16));
		return *this;
	}

	/// Put a @c uint32 in the hash stream.
	HashStream& operator<<(const uint32& value) {
		write(&value, sizeof(uint32));
		return *this;
	}

	/// Put a @c uint64 in the hash stream.
	HashStream& operator<<(const uint64& value) {
		write(&value, sizeof(uint64));
		return *this;
	}

	/// Put a @c int8 in the hash stream.
	HashStream& operator<<(const int8& value) {
		write(&value, sizeof(int8));
		return *this;
	}

	/// Put a @c int16 in the hash stream.
	HashStream& operator<<(const int16& value) {
		write(&value, sizeof(int16));
		return *this;
	}

	/// Put a @c int32 in the hash stream.
	HashStream& operator<<(const int32& value) {
		write(&value, sizeof(int32));
		return *this;
	}

	/// Put a @c int64 in the hash stream.
	HashStream& operator<<(const int64& value) {
		write(&value, sizeof(int64));
		return *this;
	}

	/// Put a @c float in the hash stream.
	HashStream& operator<<(const float& value) {
		write(&value, sizeof(float));
		return *this;
	}

	/// Put a @c double in the hash stream.
	HashStream& operator<<(const double& value) {
		write(&value, sizeof(double));
		return *this;
	}

private:
	T mBuffer;
};

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
