/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file JSONRPCBackend.h
 *    Provides JSON client and server side requests and responses.
 *
 * @author Tim Langner
 * @date   2011/02/17
 */

#ifndef _MIRA_JSONRPCBACKEND_H_
#define _MIRA_JSONRPCBACKEND_H_

#ifndef Q_MOC_RUN
#include <boost/thread/future.hpp>
#include <boost/algorithm/string.hpp>
#endif

#include <serialization/JSONSerializer.h>
#include <rpc/RPCError.h>
#include <rpc/JSONRPCResponse.h>
#include <utils/ToString.h>

#include <thread/Atomic.h>

namespace mira {

//////////////////////////////////////////////////////////////////////////////

///@cond INTERNAL

template <typename R>
inline bool jsonRPChasExceptionHelper(boost::promise<R>& promise, json::Value& value)
{
	auto it = value.get_obj().find("error");
	if(it!=value.get_obj().end()) {
		auto& obj = it->second;
		auto messageit = obj.get_obj().find("message");
		if(messageit==obj.get_obj().end()) {
			promise.set_exception(boost::copy_exception(
				XRPC("RPC client response has no correct error format")));
			return true;
		}
		promise.set_exception(boost::copy_exception(XRPC(messageit->second.get_str())));
		return true;
	}

	it = value.get_obj().find("result");
	if(it==value.get_obj().end()) {
		promise.set_exception(boost::copy_exception(
			XRPC("RPC client response has no correct result format")));
		return true;
	}
	return false;
}

template <typename R>
inline void jsonRPCreturnHelper(boost::promise<R>& promise, json::Value& value)
{
	if (jsonRPChasExceptionHelper(promise, value))
		return;
	auto it = value.get_obj().find("result");
	R ret;
	JSONDeserializer d(it->second);
	d.deserialize(ret);
	promise.set_value(ret);
}


template <>
inline void jsonRPCreturnHelper(boost::promise<JSONRPCResponse>& promise, json::Value& value)
{
	promise.set_value(JSONRPCResponse(value));
}

template <>
inline void jsonRPCreturnHelper(boost::promise<void>& promise, json::Value& value)
{
	if (jsonRPChasExceptionHelper(promise, value))
		return;
	promise.set_value();
}
///@endcond INTERNAL


/**
 * @ingroup RPCModule
 * Provides JSON client and server side requests and responses.
 */
class JSONRPCBackend
{
public:

	/**
	 * Error codes as defined by the JSON RPC standard.
	 */
	enum ErrorCode
	{
		ParseError = -32700,
		InvalidRequest = -32600,
		MethodNotFound = -32601,
		InvalidParams = -32602,
		InternalError = -32603,
		ServerError = -32099,
	};

	typedef json::Value ContainerType;

	/**
	 * @ingroup RPCModule
	 *
	 * JSON client-side response.
	 * The response must have been created by the RPCServer and it must be
	 * contained in the value that is passed in the constructor.
	 *
	 * @see @ref RPC_Backends
	 */
	class ClientResponse
	{
	public:

		ClientResponse(json::Value* value) :
			mValue(value) {}

		/**
		 * Read the response header (i.e. the call id of the RPC call)
		 * from the response. This method is called first by the RPCClient when a
		 * response is handled
		 */
		void getHeader(std::string& oCallId) const
		{
			auto it = mValue->get_obj().find("id");
			if(it== mValue->get_obj().end())
				MIRA_THROW(XRPC, "Server response is missing an 'id'. Response = " 
				           << json::write(*mValue));
			oCallId = json::write(it->second);
		}

		/**
		 * Obtain return value from response and set it using promise.set_value()
		 * or set exception using promise.set_exception().
		 * See documentation of boost::promise and boost::future for details about
		 * futures and promises.
		 * This method is called after getHeader().
		 */
		template<typename R>
		void getReturn(boost::promise<R>& promise) const
		{
			jsonRPCreturnHelper(promise, *mValue);
		}

	private:
		json::Value* mValue;
	};

	/**
	 * @ingroup RPCModule
	 *
	 * JSON client-side request.
	 * The request is written into the value that is passed in the constructor.

	 * @see @ref RPC_Backends
	 */
	class ClientRequest
	{
	public:

		ClientRequest(json::Value* value) :
			mValue(value),
			mParams(NULL)
		{
			*mValue = json::Object();
			mValue->get_obj()["jsonrpc"] = json::Value("2.0");
		}

		/**
		 * Generates a unique call ID for a client request of this backend.
		 * As call id an integer is used that is increasing with each call.
		 */
		std::string generateCallID() {
			static uint32 sID = 0;
			json::Value v(std::string("JSONCR") + toString(atomic::inc(&sID)));
			return json::write(v, false);
		}

		/**
		 * Write the request header consisting of the call ID, the service name
		 * and the signature of the requested method.
		 * This method is called first by the RCPClient when a new request is
		 * generated.
		 */
		void setHeader(const std::string& callId, const std::string& service,
					const RPCSignature& signature)
		{
			std::string method = service + "." + signature.name;
			mValue->get_obj()["method"] = json::Value(method);
			json::Value id;
			json::read(callId, id);
			mValue->get_obj()["id"] = id;
		}

		/**
		 * Write the value of the next parameter to the request.
		 * This method is called multiple times after the setHeader() method for
		 * each parameter that is passed to the RPCClient::call() method.
		 */
		template <typename P>
		void setParameter(const P& param)
		{
			// if no params inserted yet do so
			if (mParams == NULL)
			{
				mValue->get_obj()["params"] = json::Array();
				mParams = &mValue->get_obj()["params"];
			}
			mParams->get_array().push_back(mOut.serialize(param));
		}

	private:
		json::Value* mValue;
		json::Value* mParams;
		JSONSerializer mOut;
	};

	/**
	 * @ingroup RPCModule
	 *
	 * JSON server-side response.
	 * The response is written to the value that is passed in the constructor.

	 * @see @ref RPC_Backends
	 */
	class ServerResponse
	{
	public:

		ServerResponse(json::Value* value) :
			mValue(value)
		{
			*mValue = json::Object();
			mValue->get_obj()["jsonrpc"] = json::Value("2.0");
		}

		/**
		 * Write the response header consisting of the ID of the call.
		 * This method is called first by the RCPServer to create a response
		 * for a finished RCP call.
		 * It will be followed by a call of either returnException(), returnResult()
		 * or returnVoid().
		 */
		void setHeader(const std::string& callId)
		{
			json::Value id;
			json::read(callId, id);
			mValue->get_obj()["id"] = id;
		}

		/**
		 * Write exception as result an RCP call.
		 * This method is called after setHeader() in case of an error or
		 * exception while processing the call. The reason is passed
		 * as parameter.
		 */
		void returnException(RPCError reason, const std::string& message)
		{
			json::Object o;
			ErrorCode code;
			switch (reason)
			{
				case RPC_INVALID_REQUEST : code = InvalidRequest; break;
				case RPC_METHOD_NOT_FOUND : code = MethodNotFound; break;
				case RPC_INVALID_PARAMS : code = InvalidParams; break;
				case RPC_EXCEPTION_IN_CALL : code = InternalError; break;
				default : code = ServerError; break;
			}
			o["code"] = json::Value(code);
			o["message"] = json::Value(message);
			mValue->get_obj()["error"] = o;
		}

		/**
		 * Write result of an RCP call.
		 * This method is called after setHeader() in case of a successful
		 * RCP call that returns a value.
		 */
		template<typename R>
		void returnResult(const R& res)
		{
			mValue->get_obj()["result"] = mOut.serialize(res);
		}

		/**
		 * Write successful end of an RCP call.
		 * This method is called after setHeader() in case of a successful
		 * RCP call that does NOT return a value.
		 */
		void returnVoid()
		{
			mValue->get_obj()["result"] = json::Value();
		}

	private:
		json::Value* mValue;
		JSONSerializer mOut;
	};

	/**
	 * @ingroup RPCModule
	 *
	 * JSON server-side request.
	 * The request must have been created by the RPCClient and it must be
	 * contained in the buffer that is passed in the constructor.
	 *
	 * @see @ref RPC_Backends
	 */
	class ServerRequest
	{
	public:

		ServerRequest(const json::Value* value) :
			mValue(value),
			mParams(NULL),
			mIndex(0)
		{}

	public:
		/**
		 * Read the request header including the call ID, the service name
		 * and the signature of the method.
		 * The call ID and the service name must be returned via the by-reference
		 * parameters. The signature should be stored internally to be able to
		 * handle the methods below.
		 * This method is called first when a new request is handled by the RPCServer.
		 */
		void getHeader(std::string& oCallId, std::string& oService)
		{
			auto it = mValue->get_obj().find("id");
			if(it==mValue->get_obj().end())
				MIRA_THROW(XRPC,"The incoming JSON request is missing an 'id'. Request = " 
				           << json::write(*mValue));
			oCallId = json::write(it->second);

			it = mValue->get_obj().find("method");
			if(it==mValue->get_obj().end())
				MIRA_THROW(XRPC,"The incoming JSON request is missing a 'method'. Request = " 
				           << json::write(*mValue));

			std::string method = it->second.get_str();
			if(method.empty())
				MIRA_THROW(XRPC,"The incoming JSON request has an empty 'method'. Request = " 
				           << json::write(*mValue));

			std::vector<std::string> strs;
			boost::split(strs, method, boost::is_from_range('.','.'));
			oService = *strs.begin();
			mSignature.name = *strs.rbegin();

			it = mValue->get_obj().find("params");
			// method has no parameters
			if(it==mValue->get_obj().end())
				return;
			mParams = &it->second;

			for(std::size_t i=0;i<mParams->get_array().size(); ++i)
			{
				const json::Value& v = mParams->get_array()[i];
				std::string type;

				switch(v.type()) {
				case json_spirit::obj_type:   type="object"; break;
				case json_spirit::array_type: type="array"; break;
				case json_spirit::str_type:   type="string"; break;
				case json_spirit::bool_type:  type="bool"; break;
				case json_spirit::int_type:   type="int"; break;
				case json_spirit::real_type:  type="float"; break;
				default: break;
				}

				mSignature.parameterTypes.push_back(type);
			}
		}

		/**
		 * Check, if the passed signature is compatible with the signature that was
		 * read from the request header in the previous getHeader() call above.
		 */
		bool checkSignature(const RPCSignature& signature)
		{
			return signature.name == mSignature.name &&
			       signature.parameterTypes.size() == mSignature.parameterTypes.size();
		}

		/**
		 * Return the signature, that was read from the request header in the previous
		 * getHeader() call above.
		 */
		const RPCSignature& getSignature()
		{
			return mSignature;
		}

		/**
		 * Read and deserializes the next parameter from the request.
		 * This method is called multiple times for each parameter of the RPC method.
		 */
		template <typename P>
		void getParameter(P& oParam)
		{
			try
			{
				if (mParams != NULL)
				{
					assert(mIndex < (uint32)mParams->get_array().size());
					const json::Value& v = mParams->get_array()[mIndex++];
					JSONDeserializer d(v);
					// no need to do type check here, since it is done by JSONDeserializer
					d.deserialize(oParam);
				}
			}
			catch(std::exception& ex)
			{
				MIRA_THROW(XRPC,"Parameter must be of type'" << typeName<P>() << "'. " << ex.what());
			}
			catch(...)
			{
				MIRA_THROW(XRPC,"Parameter must be of type'" << typeName<P>() << "'");
			}
		}

	private:
		const json::Value* mValue;
		const json::Value* mParams;
		uint32 mIndex;
		RPCSignature mSignature;
	};

};
//////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
