/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Exceptions.h
 *    Commonly used exception classes
 *
 * @author Erik Einhorn, Tim Langner
 * @date   2010/04/01
 */

#ifndef _MIRA_EXCEPTIONS_H_
#define _MIRA_EXCEPTIONS_H_

#include <error/Exception.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup ExceptionModule
 * A class that signals an exception which can only be detected at
 * runtime, a so called runtime exception.
 * In contrast to logical exceptions (see XLogical) runtime exceptions
 * can not be detected or avoided at compile time.
 */
class XRuntime : public Exception
{
public:
	XRuntime(const std::string& msg, const char* file=NULL, int line=0) throw() :
		Exception(msg, file, line) {}
};

/**
 * @ingroup ExceptionModule
 * A exception that occurred due to an invalid configuration.
 */
class XInvalidConfig : public XRuntime
{
public:
	XInvalidConfig(const std::string& msg, const char* file=NULL, int line=0) throw() :
		XRuntime(msg, file, line) {}
};

/**
 * @ingroup ExceptionModule
 * Generic IO exception.
 */
class XIO : public XRuntime
{
public:
	XIO(const std::string& msg, const char* file=NULL, int line=0) throw() :
		XRuntime(msg, file, line) {}
};

/**
 * @ingroup ExceptionModule
 * An exception that occurred in conjunction with network connections
 */
class XConnection : public XRuntime
{
public:
	XConnection(const std::string& msg, const char* file=NULL, int line=0) throw() :
		XRuntime(msg, file, line) {}
};

/**
 * @ingroup ExceptionModule
 * An exception that occurred after a failed system call.
 */
class XSystemCall : public XRuntime
{
public:
	XSystemCall(const std::string& msg, const char* file=NULL, int line=0) throw() :
		XRuntime(msg, file, line) {}
};

/**
 * @ingroup ExceptionModule
 * An exception that indicates that a given file was not found.
 */
class XFileNotFound : public XIO
{
public:
	XFileNotFound(const std::string& msg, const char* file=NULL, int line=0) throw() :
		XIO(msg, file, line) {}
};

/**
 * @ingroup ExceptionModule
 * An exception that is thrown when a method is not supported or implemented
 */
class XNotImplemented : public XRuntime
{
public:
	XNotImplemented(const std::string& msg, const char* file=NULL, int line=0) throw() :
		XRuntime(msg, file, line) {}
};

/**
 * @ingroup ExceptionModule
 * An exception that is thrown when a timeout occurs
 */
class XTimeout : public XRuntime
{
public:
	XTimeout(const std::string& msg, const char* file=NULL, int line=0) throw() :
		XRuntime(msg, file, line) {}
};

/**
 * @ingroup ExceptionModule
 * An exception that occurred due to invalid program logic. This kind of exception
 * could be detected at design and compile time and should be fixed in the
 * final release. In bug free software only runtime exceptions should occur since
 * those cannot be avoided at compile time.
 */
class XLogical : public Exception
{
public:
	XLogical(const std::string& msg, const char* file=NULL, int line=0) throw() :
		Exception(msg, file, line) {}
};

/**
 * @ingroup ExceptionModule
 * An exception that signals invalid parameters for method calls, etc.
 */
class XInvalidParameter : public XLogical
{
public:
	XInvalidParameter(const std::string& msg, const char* file=NULL, int line=0) throw() :
		XLogical(msg, file, line) {}
};


/**
 * @ingroup ExceptionModule
 * An exception that signals an assertion, i.e. a required condition that was
 * not fulfilled. This usually is the result of a violated pre- or post-
 * condition.
 */
class XAssertion : public XLogical
{
public:
	XAssertion(const std::string& msg, const char* file=NULL, int line=0) throw() :
		XLogical(msg, file, line) {}
};

/**
 * @ingroup ExceptionModule
 * An exception that is thrown if a cast to a target type cannot be performed.
 * This usually is the result of a dynamic cast that is invoked at runtime.
 * It should be used to generate a more verbose exception than std::bad_cast
 * (which is thrown by the dynamic_cast operator as the result of a failed cast
 *  to a reference type).
 */
class XBadCast : public XRuntime
{
public:
	XBadCast(const std::string& msg, const char* file=NULL, int line=0) :
		XRuntime(msg, file, line) {}
};


/**
 * @ingroup ExceptionModule
 * An exception that occurs if an entity is accessed without  permission.
 */
class XAccessViolation : public XRuntime
{
public:
	XAccessViolation(const std::string& msg, const char* file=NULL, int line=0) throw() :
		XRuntime(msg, file, line) {}
};

/**
 * @ingroup ExceptionModule
 * An exception that occurs when allocation of requested memory fails.
 * Based on Exception and not on Runtime to match the STL exception categories.
 */
class XBadAlloc : public Exception
{
public:
	XBadAlloc(const std::string& msg, const char* file=NULL, int line=0) throw() :
		Exception(msg, file, line) {}
};

///////////////////////////////////////////////////////////////////////////////

}

#endif
